<?php
/**
 * Persian (Jalali/Shamsi) Date for WordPress
 * Converts all WordPress dates to Persian (Jalali/Shamsi) calendar
 */

// Don't run in admin area except for AJAX requests
if (is_admin() && !defined('DOING_AJAX')) {
    return;
}

// Load the JDF library if not exists
if (!function_exists('jdate')) {
    require_once get_template_directory() . '/inc/jdf.php';
}

/**
 * Convert Gregorian date to Jalali
 */
function gregorian_to_jalali_date($date) {
    if (empty($date) || is_admin()) {
        return $date;
    }
    
    // Extract date components
    $date = date_parse($date);
    if (!$date || !isset($date['year'], $date['month'], $date['day'])) {
        return $date;
    }
    
    // Convert to Jalali
    $jalali_date = gregorian_to_jalali($date['year'], $date['month'], $date['day']);
    
    // Format the date
    $formatted_date = sprintf(
        '%s/%s/%s',
        $jalali_date[0],
        str_pad($jalali_date[1], 2, '0', STR_PAD_LEFT),
        str_pad($jalali_date[2], 2, '0', STR_PAD_LEFT)
    );
    
    return $formatted_date;
}

/**
 * Convert date to Jalali format
 */
function convert_to_jalali($the_date, $format = '', $before = '', $after = '') {
    // Don't convert in admin area
    if (is_admin() && !wp_doing_ajax()) {
        return $the_date;
    }

    if (empty($the_date)) {
        return $the_date;
    }
    
    // Get the timestamp
    $timestamp = is_numeric($the_date) ? (int)$the_date : strtotime($the_date);
    
    // If timestamp is invalid, return original date
    if ($timestamp === false || $timestamp < 0) {
        return $before . $the_date . $after;
    }
    
    // If no format is provided, use WordPress date format
    if (empty($format)) {
        $format = get_option('date_format');
    }
    
    // Format the date using jdate
    $formatted_date = jdate($format, $timestamp);
    
    // If we still got an array, convert it to string
    if (is_array($formatted_date)) {
        $formatted_date = implode(' ', $formatted_date);
    }
    
    return $before . $formatted_date . $after;
}

// Only run on frontend
if (!is_admin() || (defined('DOING_AJAX') && DOING_AJAX)) {
    // Add filters for different date formats
    add_filter('get_the_date', 'convert_to_jalali', 10, 2);
    add_filter('get_comment_date', 'convert_to_jalali', 10, 2);
    add_filter('get_the_modified_date', 'convert_to_jalali', 10, 2);
    add_filter('get_the_time', 'convert_to_jalali', 10, 2);
    add_filter('get_comment_time', 'convert_to_jalali', 10, 2);
    add_filter('get_post_time', 'convert_to_jalali', 10, 2);
    add_filter('get_post_modified_time', 'convert_to_jalali', 10, 2);
    
    // For WooCommerce dates
    add_filter('woocommerce_format_datetime', 'convert_woocommerce_dates_to_jalali', 10, 2);
    
    // For ACF date fields
    add_filter('acf/format_value/type=date_picker', 'convert_acf_date_to_jalali', 10, 3);
    add_filter('acf/format_value/type=date_time_picker', 'convert_acf_date_to_jalali', 10, 3);
}

function convert_woocommerce_dates_to_jalali($date, $format) {
    if (empty($date) || is_admin()) {
        return $date;
    }
    
    $timestamp = strtotime($date);
    return $timestamp ? jdate($format, $timestamp) : $date;
}

function convert_acf_date_to_jalali($value, $post_id, $field) {
    if (empty($value) || is_admin()) {
        return $value;
    }
    
    $timestamp = strtotime($value);
    return $timestamp ? jdate('Y/m/d', $timestamp) : $value;
}

// Add Jalali date to post meta for sorting
add_action('save_post', 'add_jalali_date_meta', 10, 3);
function add_jalali_date_meta($post_id, $post, $update) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
        return;
    }
    
    if (!isset($post->post_date)) {
        return;
    }
    
    $timestamp = strtotime($post->post_date);
    if ($timestamp) {
        $jalali_date = jdate('Y-m-d H:i:s', $timestamp);
        update_post_meta($post_id, '_jalali_date', $jalali_date);
    }
}

// Initialize the Jalali date system
add_action('after_setup_theme', 'init_jalali_date_system');
function init_jalali_date_system() {
    // Set the timezone to Tehran
    date_default_timezone_set('Asia/Tehran');
    
    // Load text domain for translations
    load_theme_textdomain('jalali-date', get_template_directory() . '/languages');
}
