<?php
/**
 * JDF - Jalali (Shamsi) Date and Time Functions
 * @author JDF.SCR.IR
 * @link http://jdf.scr.ir
 */

function jdate($format, $timestamp = '', $none = '', $time_zone = 'Asia/Tehran', $tr_num = 'fa') {
    // If timestamp is empty, use current time
    if ($timestamp === '') {
        $timestamp = time();
    } elseif (!is_numeric($timestamp)) {
        $timestamp = strtotime($timestamp);
    }
    $T_sec = 0; // Timezone seconds
    
    if ($time_zone != 'local') {
        date_default_timezone_set(($time_zone == '') ? 'Asia/Tehran' : $time_zone);
    }
    
    $ts = $T_sec + (($timestamp == '' or $timestamp == 'now') ? time() : tr_num($timestamp));
    $date = explode('_', date('H_i_j_n_O_P_s_w_Y', $ts));
    
    list($j_y, $j_m, $j_d) = gregorian_to_jalali($date[8], $date[3], $date[2]);
    $doy = ($j_y % 33 % 4 - 1 == (int)($j_y % 33 * 0.05)) ? 1 : 0;
    $kab = ($j_y % 4 == 0) ? 1 : 2;
    $sl = strlen($format);
    $out = '';
    
    for ($i = 0; $i < $sl; $i++) {
        $sub = substr($format, $i, 1);
        
        if ($sub == '\\') {
            $out .= substr($format, ++$i, 1);
            continue;
        }
        
        switch ($sub) {
            case 'E':
            case 'R':
            case 'x':
            case 'X':
                $out .= 'http://jdf.scr.ir';
                break;
            
            case 'B':
            case 'e':
            case 'g':
            case 'G':
            case 'h':
            case 'I':
            case 'T':
            case 'u':
            case 'Z':
                $out .= date($sub, $ts);
                break;
            
            case 'a':
                $out .= ($date[0] < 12) ? 'ق.ظ' : 'ب.ظ';
                break;
                
            case 'A':
                $out .= ($date[0] < 12) ? 'قبل از ظهر' : 'بعد از ظهر';
                break;
            
            case 'b':
                $out .= (int)($j_m / 3.1) + 1;
                break;
                
            case 'c':
                $out .= $j_y . '/' . $j_m . '/' . $j_d . ' ' . $date[0] . ':' . $date[1] . ':' . $date[6] . ' ' . $date[5];
                break;
                
            case 'C':
                $out .= (int)(($j_y + 99) / 100);
                break;
                
            case 'd':
                $out .= ($j_d < 10) ? '0' . $j_d : $j_d;
                break;
                
            case 'D':
                $out .= jdate_words(array('kh' => $date[7]), ' ');
                break;
                
            case 'f':
                $out .= jdate_words(array('ff' => $j_m), ' ');
                break;
                
            case 'F':
                $out .= jdate_words(array('mm' => $j_m), ' ');
                break;
                
            case 'H':
                $out .= $date[0];
                break;
                
            case 'i':
                $out .= $date[1];
                break;
                
            case 'j':
                $out .= $j_d;
                break;
                
            case 'J':
                $out .= jdate_words(array('rr' => $j_d), ' ');
                break;
                
            case 'k':
                $out .= trim($date[0] == 0 ? 24 : $date[0]);
                break;
                
            case 'K':
                $out .= $date[0] / 10;
                break;
                
            case 'l':
                $out .= jdate_words(array('rh' => $date[7]), ' ');
                break;
                
            case 'L':
                $out .= $kab;
                break;
                
            case 'm':
                $out .= ($j_m < 10) ? '0' . $j_m : $j_m;
                break;
                
            case 'M':
                $out .= jdate_words(array('km' => $j_m), ' ');
                break;
                
            case 'n':
                $out .= $j_m;
                break;
                
            case 'N':
                $out .= $date[7] + 1;
                break;
                
            case 'o':
                $jdw = ($date[7] == 6) ? 0 : $date[7] + 1;
                $dny = 364 + $kab - $doy;
                $out .= ($date[7] == 6 && ($doy >= 3 || $doy - $dny >= 3) && ($dny > 3 || $doy <= 3)) ? $j_y + 1 : $j_y;
                break;
                
            case 'O':
                $out .= $date[4];
                break;
                
            case 'p':
                $out .= jdate_words(array('mb' => $j_m), ' ');
                break;
                
            case 'P':
                $out .= $date[5];
                break;
                
            case 'q':
                $out .= jdate_words(array('sh' => $j_y), ' ');
                break;
                
            case 'Q':
                $out .= $kab + 364 - $doy;
                break;
                
            case 'r':
                $key = jdate_words(array('rh' => $date[7], 'mm' => $j_m));
                $out .= $date[0] . ':' . $date[1] . ' ' . $date[4] . ' ' . $key['rh'] . ', ' . $j_d . ' ' . $key['mm'] . ' ' . $j_y;
                break;
                
            case 's':
                $out .= $date[6];
                break;
                
            case 'S':
                $out .= 'ام';
                break;
                
            case 't':
                $out .= ($j_m != 12) ? (31 - (int)($j_m / 6.5)) : ($kab + 29);
                break;
                
            case 'U':
                $out .= $ts;
                break;
                
            case 'v':
                $out .= jdate_words(array('ss' => substr($j_y, 2, 2)), ' ');
                break;
                
            case 'V':
                $out .= jdate_words(array('ss' => $j_y), ' ');
                break;
                
            case 'w':
                $out .= ($date[7] == 6) ? 0 : $date[7] + 1;
                break;
                
            case 'W':
                $avs = (($date[7] == 6) ? 0 : $date[7] + 1) - ($doy % 7);
                if ($avs < 0) $avs += 7;
                $num = (int)(($doy + $avs) / 7);
                if ($avs < 4) {
                    $num++;
                } elseif ($num < 1) {
                    $num = ($avs == 4 || $avs == (($j_y % 4 == 0) ? 4 : 3)) ? 53 : 52;
                }
                $a = ($num > 52) ? (($j_y % 4 == 0) ? 1 : 2) : 0;
                $out .= (($kab + 363 - $doy) < $a && $num == 1) ? '01' : (($num + $a < 10) ? '0' . $num + $a : $num + $a);
                break;
                
            case 'y':
                $out .= substr($j_y, 2, 2);
                break;
                
            case 'Y':
                $out .= $j_y;
                break;
                
            case 'z':
                $out .= $doy;
                break;
                
            default:
                $out .= $sub;
        }
    }
    
    // Ensure we return a string
    $result = ($tr_num != 'en') ? tr_num($out, 'fa', '') : $out;
    
    // If for some reason we still have an array, convert it to string
    if (is_array($result)) {
        $result = implode(' ', $result);
    }
    
    return $result;
}

function jdate_words($array, $mod = '') {
    $num = '';
    
    foreach ($array as $type => $number) {
        $num = (int)tr_num($number);
        
        switch ($type) {
            case 'ss':
                $sl = strlen($number);
                $xy3 = substr($number, 2 - $sl, 1);
                $h3 = $h34 = $h4 = '';
                
                if ($xy3 == 1) {
                    $p34 = '';
                    $k34 = array('ده', 'یازده', 'دوازده', 'سیزده', 'چهارده', 'پانزده', 'شانزده', 'هفده', 'هجده', 'نوزده');
                    $h34 = $k34[substr($number, 2 - $sl, 2) - 10];
                } else {
                    $xy4 = (int)substr($number, 3 - $sl, 1);
                    $p34 = ($xy3 == 0 || $xy4 == 0) ? '' : ' و ';
                    $k3 = array('', '', 'بیست', 'سی', 'چهل', 'پنجاه', 'شصت', 'هفتاد', 'هشتاد', 'نود');
                    $h3 = ($xy3 == 0) ? '' : $k3[$xy3];
                    $k4 = array('', 'یک', 'دو', 'سه', 'چهار', 'پنج', 'شش', 'هفت', 'هشت', 'نه');
                    $h4 = ($xy4 == 0) ? '' : $k4[$xy4];
                    $h34 = ($h3 == '' && $h4 == '') ? '' : $h3 . $p34 . $h4;
                }
                
                $k2 = array('هزار', 'میلیون', 'میلیارد', 'تریلیون', 'کادریلیون', 'کوینتیلیون', 'سکستیلیون', 'سپتیلیون', 'اکتیلیون', 'نانیلیون', 'دسیلیون');
                $h2 = '';
                
                if ($sl > 3) {
                    $xy2 = substr($number, 0, $sl - 3);
                    $xy2 = (int)$xy2;
                    $x2 = $xy2 % 1000;
                    $xy2 = (int)($xy2 / 1000);
                    $h2 = jdate_words(array('ss' => $xy2)) . ' ' . $k2[0];
                    
                    for ($i = 1; $i < count($k2); $i++) {
                        if (!empty($xy2)) {
                            $h2 = jdate_words(array('ss' => $xy2)) . ' ' . $k2[$i] . ' و ' . $h2;
                            $xy2 = (int)($xy2 / 1000);
                        }
                    }
                    
                    if ($x2 > 0) {
                        $h2 .= ' و ' . jdate_words(array('ss' => $x2));
                    }
                }
                
                $array[$type] = (($number > 0) ? $h2 . (($h2 != '' && $h34 != '') ? ' و ' : '') . $h34 : '') . ' ';
                break;
                
            case 'mm':
                $array[$type] = array('فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند');
                break;
                
            case 'km':
                $array[$type] = array('فر', 'ار', 'خر', 'تی', 'مر', 'شه', 'مه', 'آبا', 'آذ', 'دی', 'به', 'اس');
                break;
                
            case 'kh':
                $array[$type] = array('ی', 'د', 'س', 'چ', 'پ', 'ج', 'ش');
                break;
                
            case 'rh':
                $array[$type] = array('یکشنبه', 'دوشنبه', 'سه شنبه', 'چهارشنبه', 'پنجشنبه', 'جمعه', 'شنبه');
                break;
                
            case 'sh':
                $array[$type] = array('مار', 'اسب', 'گوسفند', 'میمون', 'مرغ', 'سگ', 'خوک', 'موش', 'گاو', 'پلنگ', 'خرگوش', 'نهنگ', 'مار', 'اسب', 'گوسفند', 'میمون', 'مرغ', 'سگ و خوک', 'موش', 'گاو', 'پلنگ', 'خرگوش', 'نهنگ');
                break;
                
            case 'mb':
                $array[$type] = array('حمل', 'ثور', 'جوزا', 'سرطان', 'اسد', 'سنبله', 'میزان', 'عقرب', 'قوس', 'جدی', 'دلو', 'حوت');
                break;
                
            case 'ff':
                $array[$type] = array('بهار', 'تابستان', 'پاییز', 'زمستان');
                break;
                
            case 'km':
                $array[$type] = array('فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند');
                break;
                
            case 'kh':
                $array[$type] = array('ی', 'د', 'س', 'چ', 'پ', 'ج', 'ش');
                break;
                
            case 'rr':
                $num = (int)$number;
                $ret = '';
                
                $k1 = array('یک', 'دو', 'سه', 'چهار', 'پنج', 'شش', 'هفت', 'هشت', 'نه');
                $k2 = array('ده', 'بیست', 'سی', 'چهل', 'پنجاه', 'شصت', 'هفتاد', 'هشتاد', 'نود');
                $k3 = array('صد', 'دویست', 'سیصد', 'چهارصد', 'پانصد', 'ششصد', 'هفتصد', 'هشتصد', 'نهصد');
                
                if ($num < 10) {
                    $ret = $k1[$num - 1];
                } elseif ($num < 20) {
                    $ret = ($num == 10) ? 'ده' : $k1[$num - 11] . 'ده';
                } elseif ($num < 100) {
                    $ret = $k2[($num / 10) - 1] . (($num % 10 > 0) ? ' و ' . jdate_words(array('rr' => $num % 10)) : '');
                } elseif ($num < 1000) {
                    $ret = $k3[($num / 100) - 1] . (($num % 100 > 0) ? ' و ' . jdate_words(array('rr' => $num % 100)) : '');
                } else {
                    $ret = jdate_words(array('rr' => (int)($num / 1000))) . ' هزار' . (($num % 1000 > 0) ? ' و ' . jdate_words(array('rr' => $num % 1000)) : '');
                }
                
                $array[$type] = $ret;
                break;
                
            default:
                $array[$type] = $number;
        }
    }
    
    return ($mod === '') ? $array : implode($mod, $array);
}

function gregorian_to_jalali($g_y, $g_m, $g_d, $mod = '') {
    $g_y = (int)$g_y;
    $g_m = (int)$g_m;
    $g_d = (int)$g_d;
    
    $d_4 = $g_y % 4;
    $g_a = array(0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334);
    $doy_g = $g_a[(int)$g_m] + $g_d;
    
    if ($d_4 == 0 && $g_m > 2) {
        $doy_g++;
    }
    
    $d_33 = (int)((($g_y - 16) % 132) * 0.0305);
    $a = ($d_33 == 3 || $d_33 < ($d_4 - 1) || $d_4 == 0) ? 286 : 287;
    $b = (($d_33 == 1 || $d_33 == 2) || ($d_33 == 3 && $d_4 == 0)) ? 78 : (($d_33 == 3 && $d_4 > 0) ? 80 : 79);
    
    if ((int)(($g_y - 10) / 63) == 30) {
        $a--;
        $b++;
    }
    
    if ($doy_g > $b) {
        $jy = $g_y - 621;
        $doy_j = $doy_g - $b;
    } else {
        $jy = $g_y - 622;
        $doy_j = $doy_g + $a;
    }
    
    if ($doy_j < 187) {
        $jm = (int)(($doy_j - 1) / 31);
        $jd = $doy_j - (31 * $jm++);
    } else {
        $jm = (int)(($doy_j - 187) / 30);
        $jd = $doy_j - 186 - ($jm * 30);
        $jm += 7;
    }
    
    return ($mod == '') ? array($jy, $jm, $jd) : $jy . $mod . $jm . $mod . $jd;
}

function jalali_to_gregorian($j_y, $j_m, $j_d, $mod = '') {
    $j_y = (int)$j_y;
    $j_m = (int)$j_m;
    $j_d = (int)$j_d;
    
    $d_4 = ($j_y + 1) % 4;
    $doy_j = ($j_m < 7) ? (($j_m - 1) * 31) + $j_d : (($j_m - 7) * 30) + $j_d + 186;
    $d_33 = (int)((($j_y - 55) % 132) * 0.0305);
    $a = ($d_33 != 3 && $d_4 <= $d_33) ? 287 : 286;
    $b = (($d_33 == 1) || ($d_33 == 2 && $d_4 == 1)) ? 78 : (($d_33 == 2 && $d_4 > 1) ? 80 : 79);
    
    if ($doy_j <= $a) {
        $gy = $j_y + 621;
        $gd = $doy_j + $b;
    } else {
        $gy = $j_y + 622;
        $gd = $doy_j - $a;
    }
    
    foreach (array(0, 31, ($gy % 4 == 0) ? 29 : 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31) as $gm => $v) {
        if ($gd <= $v) {
            break;
        }
        $gd -= $v;
    }
    
    return ($mod == '') ? array($gy, $gm, $gd) : $gy . $mod . $gm . $mod . $gd;
}

function tr_num($str, $mod = 'en', $mf = '') {
    $num_a = array('0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '.');
    $key_a = array('۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹', $mf);
    
    return ($mod == 'fa') ? str_replace($num_a, $key_a, $str) : str_replace($key_a, $num_a, $str);
}
