<?php
/**
 * Fixed version of the theme settings import function with comprehensive error tracking
 * Replace your current function with this one in theme-settings.php
 */

function mohtavanegar_import_theme_settings($import_dir) {
    error_log('========== THEME SETTINGS IMPORT START ==========');
    error_log('Function mohtavanegar_import_theme_settings called with directory: ' . $import_dir);
    
    // Initialize result
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    error_log('Import directory: ' . $import_dir);
    
    // Check if import directory exists
    if (!is_dir($import_dir)) {
        error_log('Import directory does not exist: ' . $import_dir);
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری اصلی درون‌ریزی یافت نشد.', 'mohtavanegar');
        error_log('Returning error: ' . $result['message']);
        error_log('========== THEME SETTINGS IMPORT END ==========');
        return $result;
    }
    
    // List contents of import directory for debugging
    $dir_contents = scandir($import_dir);
    error_log('Import directory contents: ' . print_r($dir_contents, true));
    
    $settings_dir = $import_dir . '/theme-settings';
    error_log('Looking for settings directory: ' . $settings_dir);
    
    // Check if settings directory exists
    if (!is_dir($settings_dir)) {
        error_log('Settings directory does not exist: ' . $settings_dir);
        
        // Try to find the theme-settings directory by scanning all subdirectories
        $found = false;
        foreach ($dir_contents as $item) {
            if ($item != '.' && $item != '..' && is_dir($import_dir . '/' . $item)) {
                $sub_dir = $import_dir . '/' . $item;
                error_log('Checking subdirectory: ' . $sub_dir);
                
                $sub_contents = scandir($sub_dir);
                if (in_array('theme-settings', $sub_contents)) {
                    $settings_dir = $sub_dir . '/theme-settings';
                    error_log('Found settings directory in subdirectory: ' . $settings_dir);
                    $found = true;
                    break;
                }
            }
        }
        
        if (!$found) {
            $result['status'] = 'error';
            $result['message'] = __('دایرکتوری تنظیمات قالب یافت نشد.', 'mohtavanegar');
            error_log('Returning error: ' . $result['message']);
            error_log('========== THEME SETTINGS IMPORT END ==========');
            return $result;
        }
    }
    
    // Import settings
    $settings_file = $settings_dir . '/settings.json';
    error_log('Looking for settings file: ' . $settings_file);
    
    if (file_exists($settings_file)) {
        error_log('Settings file found, attempting to read');
        
        $file_contents = @file_get_contents($settings_file);
        if ($file_contents === false) {
            error_log('Error reading settings file: ' . error_get_last()['message']);
            $result['status'] = 'error';
            $result['message'] = __('خطا در خواندن فایل تنظیمات.', 'mohtavanegar');
            error_log('Returning error: ' . $result['message']);
            error_log('========== THEME SETTINGS IMPORT END ==========');
            return $result;
        }
        
        error_log('Settings file contents length: ' . strlen($file_contents));
        $settings = json_decode($file_contents, true);
        
        if ($settings) {
            error_log('Settings successfully decoded, found ' . count($settings) . ' settings');
            error_log('Setting keys: ' . implode(', ', array_keys($settings)));
            
            try {
                // Update basic settings
                if (isset($settings['basic_settings'])) {
                    error_log('Updating basic_settings');
                    update_option('basic_settings', $settings['basic_settings']);
                    error_log('basic_settings updated successfully');
                }
                
                // Update header settings
                if (isset($settings['header_template'])) {
                    error_log('Updating header_template');
                    update_option('header_template', $settings['header_template']);
                    error_log('header_template updated successfully');
                }
                
                if (isset($settings['header_type'])) {
                    error_log('Updating header_type');
                    update_option('header_type', $settings['header_type']);
                    error_log('header_type updated successfully');
                }
                
                // Update footer settings
                if (isset($settings['footer_template'])) {
                    error_log('Updating footer_template');
                    update_option('footer_template', $settings['footer_template']);
                    error_log('footer_template updated successfully');
                }
                
                if (isset($settings['footer_type'])) {
                    error_log('Updating footer_type');
                    update_option('footer_type', $settings['footer_type']);
                    error_log('footer_type updated successfully');
                }
                
                // Update template settings
                if (isset($settings['single_post_template'])) {
                    error_log('Updating single_post_template');
                    update_option('single_post_template', $settings['single_post_template']);
                    error_log('single_post_template updated successfully');
                }
                
                if (isset($settings['single_product_template'])) {
                    error_log('Updating single_product_template');
                    update_option('single_product_template', $settings['single_product_template']);
                    error_log('single_product_template updated successfully');
                }
                
                // Update typography settings
                if (isset($settings['typography_settings'])) {
                    error_log('Updating typography_settings');
                    update_option('typography_settings', $settings['typography_settings']);
                    error_log('typography_settings updated successfully');
                }
                
                // Update site logo
                if (isset($settings['site_logo_url']) && !empty($settings['site_logo_url'])) {
                    error_log('Updating site logo, original URL: ' . $settings['site_logo_url']);
                    // The media importer should have already imported this image
                    // We need to find the new attachment ID by the original URL
                    try {
                        $attachment_id = mohtavanegar_get_attachment_id_by_original_url($settings['site_logo_url']);
                        error_log('Found attachment ID: ' . ($attachment_id ? $attachment_id : 'not found'));
                        
                        if ($attachment_id) {
                            update_option('site_logo', $attachment_id);
                            error_log('site_logo updated successfully');
                        } else {
                            error_log('Could not find attachment ID for logo');
                        }
                    } catch (Exception $e) {
                        error_log('Exception in attachment lookup: ' . $e->getMessage());
                    }
                }
                
                // Import theme mods
                if (isset($settings['theme_mods']) && is_array($settings['theme_mods'])) {
                    error_log('Updating theme_mods, found ' . count($settings['theme_mods']) . ' mods');
                    foreach ($settings['theme_mods'] as $key => $value) {
                        // Skip nav_menu_locations as they are handled by the menu importer
                        if ($key === 'nav_menu_locations') {
                            error_log('Skipping nav_menu_locations');
                            continue;
                        }
                        
                        error_log('Setting theme_mod: ' . $key);
                        set_theme_mod($key, $value);
                        error_log('Theme mod ' . $key . ' set successfully');
                    }
                }
                
                // Clear Elementor cache if exists
                if (function_exists('mohtavanegar_clear_elementor_cache')) {
                    error_log('Clearing Elementor cache');
                    mohtavanegar_clear_elementor_cache();
                    error_log('Elementor cache cleared successfully');
                }
                
                $result['count'] = count($settings);
                $result['message'] = __('تنظیمات قالب با موفقیت درون‌ریزی شدند.', 'mohtavanegar');
                error_log('Import successful: ' . $result['message']);
                
            } catch (Exception $e) {
                error_log('Exception caught during settings update: ' . $e->getMessage());
                error_log('Exception trace: ' . $e->getTraceAsString());
                $result['status'] = 'error';
                $result['message'] = __('خطا در به‌روزرسانی تنظیمات: ', 'mohtavanegar') . $e->getMessage();
            } catch (Error $e) {
                error_log('PHP Error caught during settings update: ' . $e->getMessage());
                error_log('Error trace: ' . $e->getTraceAsString());
                $result['status'] = 'error';
                $result['message'] = __('خطای PHP در به‌روزرسانی تنظیمات: ', 'mohtavanegar') . $e->getMessage();
            }
            
        } else {
            error_log('Failed to decode settings JSON');
            error_log('JSON error: ' . json_last_error_msg());
            $result['status'] = 'error';
            $result['message'] = __('فایل تنظیمات نامعتبر است.', 'mohtavanegar');
            error_log('Returning error: ' . $result['message']);
        }
    } else {
        error_log('Settings file not found: ' . $settings_file);
        $result['status'] = 'error';
        $result['message'] = __('فایل تنظیمات یافت نشد.', 'mohtavanegar');
        error_log('Returning error: ' . $result['message']);
    }
    
    error_log('Theme settings import completed with status: ' . $result['status']);
    error_log('Message: ' . $result['message']);
    error_log('Count: ' . $result['count']);
    error_log('========== THEME SETTINGS IMPORT END ==========');
    
    return $result;
}
