<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Theme Settings Import/Export Functions
 */

/**
 * Export theme settings
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_theme_settings($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create settings directory
    $settings_dir = $export_dir . '/theme-settings';
    if (!is_dir($settings_dir)) {
        mkdir($settings_dir, 0755, true);
    }
    
    // Get all theme settings
    $settings = array(
        'basic_settings' => get_option('basic_settings', array()),
        'header_type' => get_option('header_type', 'default'),
        'footer_type' => get_option('footer_type', 'default'),
        'single_post_template' => get_option('single_post_template', ''),
        'single_product_template' => get_option('single_product_template', ''),
        'typography_settings' => get_option('typography_settings', array()),
        'site_logo' => get_option('site_logo', ''),
        'site_logo_light' => get_option('site_logo_light', ''),
        'site_logo_dark' => get_option('site_logo_dark', ''),
        'site_logo_mobile' => get_option('site_logo_mobile', '')
    );
    
    // Get header template and store its name and conditions
    $header_template_id = get_option('header_template', '');
    if (!empty($header_template_id) && is_numeric($header_template_id)) {
        $header_template = get_post($header_template_id);
        if ($header_template) {
            $settings['header_template'] = $header_template->post_title;
            $settings['header_template_original_id'] = $header_template_id;
            $settings['header_template_conditions'] = get_post_meta($header_template_id, '_elementor_conditions', true);
            error_log('Exported header template: ' . $header_template->post_title . ' (ID: ' . $header_template_id . ')');
        }
    }
    
    // Get footer template and store its name and conditions
    $footer_template_id = get_option('footer_template', '');
    if (!empty($footer_template_id) && is_numeric($footer_template_id)) {
        $footer_template = get_post($footer_template_id);
        if ($footer_template) {
            $settings['footer_template'] = $footer_template->post_title;
            $settings['footer_template_original_id'] = $footer_template_id;
            $settings['footer_template_conditions'] = get_post_meta($footer_template_id, '_elementor_conditions', true);
            error_log('Exported footer template: ' . $footer_template->post_title . ' (ID: ' . $footer_template_id . ')');
        }
    }
    
    // Handle multiple logos
    $logo_types = array('site_logo', 'site_logo_light', 'site_logo_dark', 'site_logo_mobile');
    foreach ($logo_types as $logo_type) {
        if (!empty($settings[$logo_type])) {
            $logo_id = $settings[$logo_type];
            $logo_url = wp_get_attachment_url($logo_id);
            
            if ($logo_url) {
                $settings[$logo_type . '_url'] = $logo_url;
                
                // Get the file path of the logo
                $logo_file_path = get_attached_file($logo_id);
                
                if (file_exists($logo_file_path)) {
                    // Create logos directory if it doesn't exist
                    $logos_dir = $settings_dir . '/logos';
                    if (!is_dir($logos_dir)) {
                        mkdir($logos_dir, 0755, true);
                    }
                    
                    // Copy the logo file to the export directory
                    $file_info = pathinfo($logo_file_path);
                    $file_name = $file_info['basename'];
                    $destination = $logos_dir . '/' . $file_name;
                    
                    if (copy($logo_file_path, $destination)) {
                        error_log('Logo file copied successfully: ' . $destination);
                        
                        // Store logo file information
                        $settings[$logo_type . '_file'] = $file_name;
                        $settings[$logo_type . '_mime'] = get_post_mime_type($logo_id);
                        $settings[$logo_type . '_title'] = get_the_title($logo_id);
                        $settings[$logo_type . '_alt'] = get_post_meta($logo_id, '_wp_attachment_image_alt', true);
                        $settings[$logo_type . '_meta'] = wp_get_attachment_metadata($logo_id);
                    }
                }
            }
        }
    }
    
    // Export typography settings with additional data
    if (!empty($settings['typography_settings'])) {
        $typography = $settings['typography_settings'];
        
        // Add font files if custom fonts are used
        if (!empty($typography['custom_fonts'])) {
            $fonts_dir = $settings_dir . '/fonts';
            if (!is_dir($fonts_dir)) {
                mkdir($fonts_dir, 0755, true);
            }
            
            foreach ($typography['custom_fonts'] as $font) {
                if (!empty($font['file'])) {
                    $font_path = get_template_directory() . '/assets/fonts/' . $font['file'];
                    if (file_exists($font_path)) {
                        copy($font_path, $fonts_dir . '/' . $font['file']);
                    }
                }
            }
        }
        
        // Add font weights and sizes
        $settings['typography_settings']['font_weights'] = get_option('mohtavanegar_font_weights', array());
        $settings['typography_settings']['font_sizes'] = get_option('mohtavanegar_font_sizes', array());
    }
    
    // Export customizer settings
    $theme_mods = get_theme_mods();
    if ($theme_mods) {
        $settings['theme_mods'] = $theme_mods;
    }
    
    // Save settings to file
    $settings_file = $settings_dir . '/settings.json';
    file_put_contents($settings_file, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($settings);
    $result['message'] = __('تنظیمات قالب با موفقیت برون‌ریزی شدند.', 'mohtavanegar');
    
    return $result;
}

/**
 * Import theme settings
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_theme_settings($import_dir) {
    error_log('========== THEME SETTINGS IMPORT START ==========');
    error_log('Function mohtavanegar_import_theme_settings called with directory: ' . $import_dir);
    
    // Initialize result
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    error_log('Import directory: ' . $import_dir);
    
    // Check if import directory exists
    if (!is_dir($import_dir)) {
        error_log('Import directory does not exist: ' . $import_dir);
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری اصلی درون‌ریزی یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // List contents of import directory for debugging
    $dir_contents = scandir($import_dir);
    error_log('Import directory contents: ' . print_r($dir_contents, true));
    
    $settings_dir = $import_dir . '/theme-settings';
    error_log('Looking for settings directory: ' . $settings_dir);
    
    // Check if settings directory exists
    if (!is_dir($settings_dir)) {
        error_log('Settings directory does not exist: ' . $settings_dir);
        
        // Try to find the theme-settings directory by scanning all subdirectories
        $found = false;
        foreach ($dir_contents as $item) {
            if ($item != '.' && $item != '..' && is_dir($import_dir . '/' . $item)) {
                $sub_dir = $import_dir . '/' . $item;
                error_log('Checking subdirectory: ' . $sub_dir);
                
                $sub_contents = scandir($sub_dir);
                if (in_array('theme-settings', $sub_contents)) {
                    $settings_dir = $sub_dir . '/theme-settings';
                    error_log('Found settings directory in subdirectory: ' . $settings_dir);
                    $found = true;
                    break;
                }
            }
        }
        
        if (!$found) {
            $result['status'] = 'error';
            $result['message'] = __('دایرکتوری تنظیمات قالب یافت نشد.', 'mohtavanegar');
            return $result;
        }
    }
    
    // Import settings
    $settings_file = $settings_dir . '/settings.json';
    error_log('Looking for settings file: ' . $settings_file);
    
    if (file_exists($settings_file)) {
        error_log('Settings file found, attempting to read');
        
        $file_contents = @file_get_contents($settings_file);
        if ($file_contents === false) {
            error_log('Error reading settings file: ' . error_get_last()['message']);
            $result['status'] = 'error';
            $result['message'] = __('خطا در خواندن فایل تنظیمات.', 'mohtavanegar');
            return $result;
        }
        
        error_log('Settings file contents length: ' . strlen($file_contents));
        $settings = json_decode($file_contents, true);
        
        if ($settings) {
            error_log('Settings successfully decoded, found ' . count($settings) . ' settings');
            error_log('Setting keys: ' . implode(', ', array_keys($settings)));
            
            try {
                // Update basic settings
                if (isset($settings['basic_settings'])) {
                    error_log('Updating basic_settings');
                    update_option('basic_settings', $settings['basic_settings']);
                    error_log('basic_settings updated successfully');
                }
                
                // Update header settings with conditions
                if (isset($settings['header_template']) && !empty($settings['header_template'])) {
                    error_log('Updating header_template: ' . $settings['header_template']);
                    
                    // Check if we have a template name instead of ID
                    if (!is_numeric($settings['header_template'])) {
                        // Try to find the template by name
                        $template_name = $settings['header_template'];
                        error_log('Looking for header template by name: ' . $template_name);
                        
                        $args = array(
                            'post_type' => 'elementor_library',
                            'post_status' => 'publish',
                            'posts_per_page' => 1,
                            'title' => $template_name,
                            'exact_title' => true,
                        );
                        
                        $template_query = new WP_Query($args);
                        
                        if ($template_query->have_posts()) {
                            $template = $template_query->posts[0];
                            $template_id = $template->ID;
                            error_log('Found header template: ' . $template_name . ' with ID: ' . $template_id);
                            update_option('header_template', $template_id);
                            error_log('header_template updated with ID: ' . $template_id);
                            
                            // Update template conditions if available
                            if (isset($settings['header_template_conditions'])) {
                                update_post_meta($template_id, '_elementor_conditions', $settings['header_template_conditions']);
                            }
                        } else {
                            error_log('Header template not found by name: ' . $template_name);
                            
                            // If we have the original ID, try to use it as fallback
                            if (isset($settings['header_template_original_id']) && !empty($settings['header_template_original_id'])) {
                                $original_id = intval($settings['header_template_original_id']);
                                error_log('Using original header template ID as fallback: ' . $original_id);
                                update_option('header_template', $original_id);
                            } else {
                                // If all else fails, just store the name as is
                                update_option('header_template', $template_name);
                                error_log('Stored header template name as is: ' . $template_name);
                            }
                        }
                    } else {
                        // It's already a numeric ID, just use it
                        $template_id = intval($settings['header_template']);
                        update_option('header_template', $template_id);
                        error_log('header_template updated with numeric ID: ' . $template_id);
                    }
                    
                    error_log('header_template update completed');
                }
                
                if (isset($settings['header_type'])) {
                    error_log('Updating header_type');
                    update_option('header_type', $settings['header_type']);
                    error_log('header_type updated successfully');
                }
                
                // Update footer settings with conditions
                if (isset($settings['footer_template']) && !empty($settings['footer_template'])) {
                    error_log('Updating footer_template: ' . $settings['footer_template']);
                    
                    // Check if we have a template name instead of ID
                    if (!is_numeric($settings['footer_template'])) {
                        // Try to find the template by name
                        $template_name = $settings['footer_template'];
                        error_log('Looking for footer template by name: ' . $template_name);
                        
                        $args = array(
                            'post_type' => 'elementor_library',
                            'post_status' => 'publish',
                            'posts_per_page' => 1,
                            'title' => $template_name,
                            'exact_title' => true,
                        );
                        
                        $template_query = new WP_Query($args);
                        
                        if ($template_query->have_posts()) {
                            $template = $template_query->posts[0];
                            $template_id = $template->ID;
                            error_log('Found footer template: ' . $template_name . ' with ID: ' . $template_id);
                            update_option('footer_template', $template_id);
                            error_log('footer_template updated with ID: ' . $template_id);
                            
                            // Update template conditions if available
                            if (isset($settings['footer_template_conditions'])) {
                                update_post_meta($template_id, '_elementor_conditions', $settings['footer_template_conditions']);
                            }
                        } else {
                            error_log('Footer template not found by name: ' . $template_name);
                            
                            // If we have the original ID, try to use it as fallback
                            if (isset($settings['footer_template_original_id']) && !empty($settings['footer_template_original_id'])) {
                                $original_id = intval($settings['footer_template_original_id']);
                                error_log('Using original footer template ID as fallback: ' . $original_id);
                                update_option('footer_template', $original_id);
                            } else {
                                // If all else fails, just store the name as is
                                update_option('footer_template', $template_name);
                                error_log('Stored footer template name as is: ' . $template_name);
                            }
                        }
                    } else {
                        // It's already a numeric ID, just use it
                        $template_id = intval($settings['footer_template']);
                        update_option('footer_template', $template_id);
                        error_log('footer_template updated with numeric ID: ' . $template_id);
                    }
                    
                    error_log('footer_template update completed');
                }
            
                if (isset($settings['footer_type'])) {
                    error_log('Updating footer_type');
                    update_option('footer_type', $settings['footer_type']);
                    error_log('footer_type updated successfully');
                }
                
                // Update template settings
                if (isset($settings['single_post_template'])) {
                    error_log('Updating single_post_template');
                    update_option('single_post_template', $settings['single_post_template']);
                    error_log('single_post_template updated successfully');
                }
                
                if (isset($settings['single_product_template'])) {
                    error_log('Updating single_product_template');
                    update_option('single_product_template', $settings['single_product_template']);
                    error_log('single_product_template updated successfully');
                }
                
                // Handle multiple logos
                $logo_types = array('site_logo', 'site_logo_light', 'site_logo_dark', 'site_logo_mobile');
                foreach ($logo_types as $logo_type) {
                    if (isset($settings[$logo_type . '_file']) && !empty($settings[$logo_type . '_file'])) {
                        $logos_dir = $settings_dir . '/logos';
                        $logo_file_path = $logos_dir . '/' . $settings[$logo_type . '_file'];
                        
                        if (file_exists($logo_file_path)) {
                            error_log('Logo file exists: ' . $logo_file_path);
                            
                            // Get WordPress upload directory
                            $upload_dir = wp_upload_dir();
                            $year_month = date('Y/m');
                            $destination_dir = $upload_dir['basedir'] . '/' . $year_month;
                            
                            // Create directory if it doesn't exist
                            if (!is_dir($destination_dir)) {
                                wp_mkdir_p($destination_dir);
                            }
                            
                            $destination_file = $destination_dir . '/' . $settings[$logo_type . '_file'];
                            
                            // Copy logo file to uploads directory
                            if (copy($logo_file_path, $destination_file)) {
                                error_log('Logo file copied to: ' . $destination_file);
                                
                                // Prepare attachment data
                                $file_type = wp_check_filetype($settings[$logo_type . '_file'], null);
                                $mime_type = isset($settings[$logo_type . '_mime']) ? $settings[$logo_type . '_mime'] : $file_type['type'];
                                $title = isset($settings[$logo_type . '_title']) ? $settings[$logo_type . '_title'] : __('Site Logo', 'mohtavanegar');
                                
                                $attachment = array(
                                    'guid' => $upload_dir['baseurl'] . '/' . $year_month . '/' . $settings[$logo_type . '_file'],
                                    'post_mime_type' => $mime_type,
                                    'post_title' => $title,
                                    'post_content' => '',
                                    'post_status' => 'inherit'
                                );
                                
                                // Insert attachment
                                $attachment_id = wp_insert_attachment($attachment, $destination_file);
                                
                                if (!is_wp_error($attachment_id)) {
                                    error_log('Logo attachment created with ID: ' . $attachment_id);
                                    
                                    // Generate attachment metadata
                                    require_once(ABSPATH . 'wp-admin/includes/image.php');
                                    $attachment_data = wp_generate_attachment_metadata($attachment_id, $destination_file);
                                    wp_update_attachment_metadata($attachment_id, $attachment_data);
                                    
                                    // Update alt text if available
                                    if (isset($settings[$logo_type . '_alt'])) {
                                        update_post_meta($attachment_id, '_wp_attachment_image_alt', $settings[$logo_type . '_alt']);
                                    }
                                    
                                    // Store original URL for reference
                                    if (isset($settings[$logo_type . '_url'])) {
                                        update_post_meta($attachment_id, '_mohtavanegar_original_url', $settings[$logo_type . '_url']);
                                    }
                                    
                                    // Update site logo option
                                    update_option($logo_type, $attachment_id);
                                    set_theme_mod('custom_logo', $attachment_id);
                                    
                                    error_log('site_logo and custom_logo updated with new attachment ID: ' . $attachment_id);
                                } else {
                                    error_log('Error creating logo attachment: ' . $attachment_id->get_error_message());
                                }
                            } else {
                                error_log('Failed to copy logo file to uploads directory');
                            }
                        } else {
                            error_log('Logo file does not exist in import directory: ' . $logo_file_path);
                        }
                    }
                }
                
                // Update typography settings
                if (isset($settings['typography_settings'])) {
                    $typography = $settings['typography_settings'];
                    
                    // Import custom fonts if available
                    if (!empty($typography['custom_fonts'])) {
                        $fonts_dir = get_template_directory() . '/assets/fonts';
                        if (!is_dir($fonts_dir)) {
                            wp_mkdir_p($fonts_dir);
                        }
                        
                        $import_fonts_dir = $settings_dir . '/fonts';
                        if (is_dir($import_fonts_dir)) {
                            foreach ($typography['custom_fonts'] as $font) {
                                if (!empty($font['file'])) {
                                    $font_path = $import_fonts_dir . '/' . $font['file'];
                                    if (file_exists($font_path)) {
                                        copy($font_path, $fonts_dir . '/' . $font['file']);
                                    }
                                }
                            }
                        }
                    }
                    
                    // Update font weights and sizes
                    if (isset($typography['font_weights'])) {
                        update_option('mohtavanegar_font_weights', $typography['font_weights']);
                    }
                    
                    if (isset($typography['font_sizes'])) {
                        update_option('mohtavanegar_font_sizes', $typography['font_sizes']);
                    }
                    
                    update_option('typography_settings', $typography);
                }
                
                // Import theme mods
                if (isset($settings['theme_mods']) && is_array($settings['theme_mods'])) {
                    error_log('Updating theme_mods, found ' . count($settings['theme_mods']) . ' mods');
                    foreach ($settings['theme_mods'] as $key => $value) {
                        // Skip nav_menu_locations as they are handled by the menu importer
                        if ($key === 'nav_menu_locations') {
                            error_log('Skipping nav_menu_locations');
                            continue;
                        }
                        
                        error_log('Setting theme_mod: ' . $key);
                        set_theme_mod($key, $value);
                        error_log('Theme mod ' . $key . ' set successfully');
                    }
                }
                
                // Clear Elementor cache if exists - direct method to avoid security checks
                if (class_exists('\Elementor\Plugin')) {
                    error_log('Clearing Elementor cache directly');
                    try {
                        \Elementor\Plugin::$instance->files_manager->clear_cache();
                        error_log('Elementor cache cleared successfully');
                    } catch (Exception $e) {
                        error_log('Error clearing Elementor cache: ' . $e->getMessage());
                    }
                } else {
                    error_log('Elementor class not found, skipping cache clearing');
                }
                
                $result['count'] = count($settings);
                $result['message'] = __('تنظیمات قالب با موفقیت درون‌ریزی شدند.', 'mohtavanegar');
                error_log('Import successful: ' . $result['message']);
            } catch (Exception $e) {
                error_log('Exception caught during settings update: ' . $e->getMessage());
                error_log('Exception trace: ' . $e->getTraceAsString());
                $result['status'] = 'error';
                $result['message'] = __('خطا در به‌روزرسانی تنظیمات: ', 'mohtavanegar') . $e->getMessage();
            } catch (Error $e) {
                error_log('PHP Error caught during settings update: ' . $e->getMessage());
                error_log('Error trace: ' . $e->getTraceAsString());
                $result['status'] = 'error';
                $result['message'] = __('خطای PHP در به‌روزرسانی تنظیمات: ', 'mohtavanegar') . $e->getMessage();
            }
        } else {
            $result['status'] = 'error';
            $result['message'] = __('فایل تنظیمات نامعتبر است.', 'mohtavanegar');
        }
    } else {
        $result['status'] = 'error';
        $result['message'] = __('فایل تنظیمات یافت نشد.', 'mohtavanegar');
    }
    
    return $result;
}
