jQuery(document).ready(function($) {
    console.log('Import/Export JS loaded');
    console.log('mohtavanegarImportExport:', mohtavanegarImportExport);
    // Tab switching
    $('.mohtavanegar-tab').on('click', function() {
        const tabId = $(this).data('tab');
        
        // Update active tab
        $('.mohtavanegar-tab').removeClass('active');
        $(this).addClass('active');
        
        // Update active content
        $('.mohtavanegar-tab-content').removeClass('active');
        $('#' + tabId + '-tab').addClass('active');
    });
    
    // Export button click
    $('#export-button').on('click', function(e) {
        e.preventDefault();
        console.log('Export button clicked');
        
        const exportItems = [];
        $('input[name="export_items[]"]:checked').each(function() {
            exportItems.push($(this).val());
        });
        
        if (exportItems.length === 0) {
            alert('لطفاً حداقل یک مورد را برای برون‌ریزی انتخاب کنید.');
            return;
        }
        
        // Show progress container
        $('.export-progress-container').show();
        $('.export-progress-bar').css('width', '0%');
        $('.export-progress-text').text(mohtavanegarImportExport.export_text);
        
        // Start export process
        startExport(exportItems);
    });
    
    // Import button click
    $('#import-button').on('click', function(e) {
        e.preventDefault();
        console.log('Import button clicked');
        
        const importFile = $('#import_file')[0].files[0];
        console.log('Import file:', importFile);
        if (!importFile) {
            alert('لطفاً یک فایل برای درون‌ریزی انتخاب کنید.');
            return;
        }
        
        const importItems = [];
        $('input[name="import_items[]"]:checked').each(function() {
            importItems.push($(this).val());
        });
        
        if (importItems.length === 0) {
            alert('لطفاً حداقل یک مورد را برای درون‌ریزی انتخاب کنید.');
            return;
        }
        
        // Show progress container
        $('.import-progress-container').show();
        $('.import-progress-bar').css('width', '0%');
        $('.import-progress-text').text(mohtavanegarImportExport.import_text);
        
        // Start import process
        startImport(importFile, importItems);
    });
    
    /**
     * Start the export process
     * 
     * @param {Array} exportItems Items to export
     */
    function startExport(exportItems) {
        // Initialize export
        $.ajax({
            url: mohtavanegarImportExport.ajaxurl,
            type: 'POST',
            data: {
                action: 'mohtavanegar_start_export',
                nonce: mohtavanegarImportExport.nonce,
                export_items: exportItems
            },
            success: function(response) {
                if (response.success) {
                    // Update progress
                    updateExportProgress(response.data.export_id, 0);
                } else {
                    handleExportError(response.data.message);
                }
            },
            error: function() {
                handleExportError('خطا در ارتباط با سرور.');
            }
        });
    }
    
    /**
     * Update export progress
     * 
     * @param {string} exportId Export ID
     * @param {number} step Current step
     */
    function updateExportProgress(exportId, step) {
        $.ajax({
            url: mohtavanegarImportExport.ajaxurl,
            type: 'POST',
            data: {
                action: 'mohtavanegar_export_progress',
                nonce: mohtavanegarImportExport.nonce,
                export_id: exportId,
                step: step
            },
            success: function(response) {
                if (response.success) {
                    const progress = response.data.progress;
                    const currentItem = response.data.current_item;
                    const isComplete = response.data.is_complete;
                    const downloadUrl = response.data.download_url;
                    
                    // Update progress bar
                    $('.export-progress-bar').css('width', progress + '%');
                    $('.export-progress-text').text(currentItem + ' - ' + progress + '%');
                    
                    if (isComplete) {
                        $('.export-progress-text').text(mohtavanegarImportExport.complete_text);
                        
                        // Create download button
                        const downloadButton = $('<a>', {
                            'href': downloadUrl,
                            'class': 'button button-primary',
                            'text': 'دانلود فایل برون‌ریزی',
                            'download': ''
                        });
                        
                        $('.export-actions').html('').append(downloadButton);
                    } else {
                        // Continue with next step
                        setTimeout(function() {
                            updateExportProgress(exportId, step + 1);
                        }, 500);
                    }
                } else {
                    handleExportError(response.data.message);
                }
            },
            error: function() {
                handleExportError('خطا در ارتباط با سرور.');
            }
        });
    }
    
    /**
     * Handle export error
     * 
     * @param {string} message Error message
     */
    function handleExportError(message) {
        $('.export-progress-text').text(mohtavanegarImportExport.error_text + ': ' + message);
        $('.export-progress-bar').css('background-color', '#dc3232');
    }
    
    /**
     * Start the import process
     * 
     * @param {File} importFile Import file
     * @param {Array} importItems Items to import
     */
    function startImport(importFile, importItems) {
        console.log('Starting import process...');
        console.log('Import items:', importItems);
        console.log('Import file name:', importFile.name);
        console.log('Import file size:', importFile.size);
        
        const formData = new FormData();
        formData.append('action', 'mohtavanegar_start_import');
        formData.append('nonce', mohtavanegarImportExport.nonce);
        formData.append('import_file', importFile);
        formData.append('import_items', JSON.stringify(importItems));
        
        $.ajax({
            url: mohtavanegarImportExport.ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                console.log('Import AJAX response:', response);
                if (response.success && response.data && response.data.import_id) {
                    // Update progress
                    console.log('Import started successfully, ID:', response.data.import_id);
                    updateImportProgress(response.data.import_id, 0);
                } else if (response.success) {
                    console.error('Import ID missing in response');
                    handleImportError('شناسه درون‌ریزی یافت نشد. لطفاً دوباره تلاش کنید.');
                } else {
                    console.error('Import error:', response.data ? response.data.message : 'Unknown error');
                    handleImportError(response.data && response.data.message ? response.data.message : 'خطای نامشخص در درون‌ریزی');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error:', status, error);
                console.error('Response text:', xhr.responseText);
                handleImportError('خطا در ارتباط با سرور: ' + error);
            }
        });
    }
    
    /**
     * Update import progress
     * 
     * @param {string} importId Import ID
     * @param {number} step Current step
     */
    function updateImportProgress(importId, step) {
        console.log('Updating import progress, ID:', importId, 'Step:', step);
        console.log('Using nonce:', mohtavanegarImportExport.nonce);
        
        // Ensure we have a valid import ID
        if (!importId) {
            console.error('Invalid import ID');
            handleImportError('شناسه درون‌ریزی نامعتبر است.');
            return;
        }
        
        $.ajax({
            url: mohtavanegarImportExport.ajaxurl,
            type: 'POST',
            data: {
                action: 'mohtavanegar_import_progress',
                nonce: mohtavanegarImportExport.nonce,
                import_id: importId,
                step: step
            },
            success: function(response) {
                if (response.success) {
                    const progress = response.data.progress;
                    const currentItem = response.data.current_item;
                    const isComplete = response.data.is_complete;
                    
                    // Update progress bar
                    $('.import-progress-bar').css('width', progress + '%');
                    $('.import-progress-text').text(currentItem + ' - ' + progress + '%');
                    
                    if (isComplete) {
                        $('.import-progress-text').text(mohtavanegarImportExport.complete_text);
                        
                        // Show success message
                        const successMessage = $('<div>', {
                            'class': 'status-message status-success',
                            'text': 'درون‌ریزی با موفقیت انجام شد.',
                            'style': 'display: block;'
                        });
                        
                        $('.import-actions').html('').append(successMessage);
                        
                        // Reload page after 3 seconds
                        setTimeout(function() {
                            location.reload();
                        }, 3000);
                    } else {
                        // Continue with next step
                        setTimeout(function() {
                            updateImportProgress(importId, step + 1);
                        }, 500);
                    }
                } else {
                    handleImportError(response.data.message);
                }
            },
            error: function() {
                handleImportError('خطا در ارتباط با سرور.');
            }
        });
    }
    
    /**
     * Handle import error
     * 
     * @param {string} message Error message
     */
    function handleImportError(message) {
        $('.import-progress-text').text(mohtavanegarImportExport.error_text + ': ' + message);
        $('.import-progress-bar').css('background-color', '#dc3232');
    }
});
