<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Register Q&A Custom Post Type and Admin Functionality
 */
class Mohtavanegar_QA_System {
    
    public function __construct() {
        // Register post type
        add_action('init', array($this, 'register_qa_post_type'));
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_qa_admin_menu'));
        
        // Register AJAX handlers for import/export
        add_action('wp_ajax_export_qa_data', array($this, 'export_qa_data'));
        add_action('wp_ajax_import_qa_data', array($this, 'import_qa_data'));
        
        // Register AJAX handlers for admin
        add_action('wp_ajax_approve_qa_comment', array($this, 'approve_qa_comment'));
        add_action('wp_ajax_delete_qa_comment', array($this, 'delete_qa_comment'));
        add_action('wp_ajax_save_qa_answer', array($this, 'save_qa_answer'));
        
        // Register AJAX handler for frontend submission
        add_action('wp_ajax_submit_qa_comment', array($this, 'submit_qa_comment'));
        add_action('wp_ajax_nopriv_submit_qa_comment', array($this, 'submit_qa_comment'));
        
        // Add admin scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Register the Q&A custom post type
     */
    public function register_qa_post_type() {
        $labels = array(
            'name'               => esc_html__('پرسش و پاسخ ها', 'mohtavanegar'),
            'singular_name'      => esc_html__('پرسش و پاسخ', 'mohtavanegar'),
            'menu_name'          => esc_html__('پرسش و پاسخ ها', 'mohtavanegar'),
            'add_new'            => esc_html__('افزودن جدید', 'mohtavanegar'),
            'add_new_item'       => esc_html__('افزودن پرسش و پاسخ جدید', 'mohtavanegar'),
            'edit_item'          => esc_html__('ویرایش پرسش و پاسخ', 'mohtavanegar'),
            'new_item'           => esc_html__('پرسش و پاسخ جدید', 'mohtavanegar'),
            'view_item'          => esc_html__('مشاهده پرسش و پاسخ', 'mohtavanegar'),
            'search_items'       => esc_html__('جستجوی پرسش و پاسخ ها', 'mohtavanegar'),
            'not_found'          => esc_html__('پرسش و پاسخی یافت نشد', 'mohtavanegar'),
            'not_found_in_trash' => esc_html__('پرسش و پاسخی در سطل زباله یافت نشد', 'mohtavanegar'),
        );
        
        $args = array(
            'labels'              => $labels,
            'public'              => false,
            'show_ui'             => true,
            'show_in_menu'        => false, // We'll add a custom menu
            'capability_type'     => 'post',
            'hierarchical'        => false,
            'supports'            => array('title', 'editor', 'author'),
            'has_archive'         => false,
            'menu_icon'           => 'dashicons-format-chat',
        );
        
        register_post_type('qa_comment', $args);
    }
    
    /**
     * Add admin menu for Q&A
     */
    public function add_qa_admin_menu() {
        add_menu_page(
            esc_html__('پرسش و پاسخ ها', 'mohtavanegar'),
            esc_html__('پرسش و پاسخ ها', 'mohtavanegar'),
            'manage_options',
            'qa-comments',
            array($this, 'render_qa_admin_page'),
            'dashicons-format-chat',
            25
        );
    }
    
    /**
     * Render the admin page for Q&A comments
     */
    public function render_qa_admin_page() {
        // Get all comments
        $pending_comments = $this->get_qa_comments('pending');
        $approved_comments = $this->get_qa_comments('publish');
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('مدیریت پرسش و پاسخ ها', 'mohtavanegar'); ?></h1>
            
            <div class="qa-import-export-section" style="margin: 20px 0; padding: 15px; background: #fff; border: 1px solid #ccd0d4; border-radius: 4px;">
                <h2><?php echo esc_html__('درون‌ریزی و برون‌ریزی', 'mohtavanegar'); ?></h2>
                <p>
                    <button id="export-qa-data" class="button button-secondary"><?php echo esc_html__('برون‌ریزی داده‌ها (JSON)', 'mohtavanegar'); ?></button>
                </p>
                
                <div class="import-section" style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #eee;">
                    <h3><?php echo esc_html__('درون‌ریزی داده‌ها', 'mohtavanegar'); ?></h3>
                    <div class="import-controls" style="margin-bottom: 15px;">
                        <label for="import-service-select"><?php echo esc_html__('انتخاب خدمت برای درون‌ریزی:', 'mohtavanegar'); ?></label>
                        <select id="import-service-select" style="margin-left: 10px; min-width: 200px;">
                            <option value=""><?php echo esc_html__('-- بدون خدمت --', 'mohtavanegar'); ?></option>
                            <?php
                            $services = get_posts(array(
                                'post_type' => 'services',
                                'posts_per_page' => -1,
                                'orderby' => 'title',
                                'order' => 'ASC',
                                'post_status' => 'publish'
                            ));
                            foreach ($services as $service) {
                                echo '<option value="' . esc_attr($service->ID) . '">' . esc_html($service->post_title) . '</option>';
                            }
                            ?>
                        </select>
                    </div>
                    <p>
                        <input type="file" id="import-qa-file" accept=".json" style="display: none;">
                        <button id="import-qa-data" class="button button-primary"><?php echo esc_html__('درون‌ریزی داده‌ها (JSON)', 'mohtavanegar'); ?></button>
                    </p>
                </div>
                <div id="import-export-message" style="margin-top: 10px;"></div>
            </div>
            
            <h2 class="nav-tab-wrapper">
                <a href="#pending-comments" class="nav-tab nav-tab-active"><?php echo esc_html__('نظرات در انتظار تایید', 'mohtavanegar'); ?></a>
                <a href="#approved-comments" class="nav-tab"><?php echo esc_html__('نظرات تایید شده', 'mohtavanegar'); ?></a>
            </h2>
            
            <div id="pending-comments" class="tab-content">
                <h3><?php echo esc_html__('نظرات در انتظار تایید', 'mohtavanegar'); ?></h3>
                
                <?php if (empty($pending_comments)) : ?>
                    <p><?php echo esc_html__('هیچ نظری در انتظار تایید وجود ندارد.', 'mohtavanegar'); ?></p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('نام', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('ایمیل', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('موبایل', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('خدمت', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('پرسش', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('تاریخ', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('عملیات', 'mohtavanegar'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pending_comments as $comment) : ?>
                                <tr id="comment-<?php echo esc_attr($comment->ID); ?>">
                                    <td><?php echo esc_html(get_post_meta($comment->ID, '_qa_name', true)); ?></td>
                                    <td><?php echo esc_html(get_post_meta($comment->ID, '_qa_email', true)); ?></td>
                                    <td><?php echo esc_html(get_post_meta($comment->ID, '_qa_mobile', true)); ?></td>
                                    <td>
                                        <?php
                                        $service_id = get_post_meta($comment->ID, '_qa_service', true);
                                        if ($service_id) {
                                            $service = get_post($service_id);
                                            echo esc_html($service ? $service->post_title : '-');
                                        } else {
                                            echo '-';
                                        }
                                        ?>
                                    </td>
                                    <td><?php echo esc_html($comment->post_title); ?></td>
                                    <td><?php echo get_the_date('Y-m-d H:i:s', $comment->ID); ?></td>
                                    <td>
                                        <button class="button approve-comment" data-id="<?php echo esc_attr($comment->ID); ?>"><?php echo esc_html__('تایید', 'mohtavanegar'); ?></button>
                                        <button class="button answer-comment" data-id="<?php echo esc_attr($comment->ID); ?>"><?php echo esc_html__('پاسخ', 'mohtavanegar'); ?></button>
                                        <button class="button delete-comment" data-id="<?php echo esc_attr($comment->ID); ?>"><?php echo esc_html__('حذف', 'mohtavanegar'); ?></button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
            
            <div id="approved-comments" class="tab-content" style="display: none;">
                <h3><?php echo esc_html__('نظرات تایید شده', 'mohtavanegar'); ?></h3>
                
                <?php if (empty($approved_comments)) : ?>
                    <p><?php echo esc_html__('هیچ نظر تایید شده‌ای وجود ندارد.', 'mohtavanegar'); ?></p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('نام', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('ایمیل', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('موبایل', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('خدمت', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('پرسش', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('پاسخ', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('تاریخ', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('عملیات', 'mohtavanegar'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($approved_comments as $comment) : ?>
                                <tr id="comment-<?php echo esc_attr($comment->ID); ?>">
                                    <td><?php echo esc_html(get_post_meta($comment->ID, '_qa_name', true)); ?></td>
                                    <td><?php echo esc_html(get_post_meta($comment->ID, '_qa_email', true)); ?></td>
                                    <td><?php echo esc_html(get_post_meta($comment->ID, '_qa_mobile', true)); ?></td>
                                    <td>
                                        <?php
                                        $service_id = get_post_meta($comment->ID, '_qa_service', true);
                                        if ($service_id) {
                                            $service = get_post($service_id);
                                            echo esc_html($service ? $service->post_title : '-');
                                        } else {
                                            echo '-';
                                        }
                                        ?>
                                    </td>
                                    <td><?php echo esc_html($comment->post_title); ?></td>
                                    <td><?php echo wp_kses_post(get_post_meta($comment->ID, '_qa_answer', true)); ?></td>
                                    <td><?php echo get_the_date('Y-m-d H:i:s', $comment->ID); ?></td>
                                    <td>
                                        <button class="button answer-comment" data-id="<?php echo esc_attr($comment->ID); ?>"><?php echo esc_html__('پاسخ', 'mohtavanegar'); ?></button>
                                        <button class="button delete-comment" data-id="<?php echo esc_attr($comment->ID); ?>"><?php echo esc_html__('حذف', 'mohtavanegar'); ?></button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get Q&A comments by status
     */
    private function get_qa_comments($status = 'pending') {
        $args = array(
            'post_type'      => 'qa_comment',
            'post_status'    => $status,
            'posts_per_page' => -1,
            'orderby'        => 'date',
            'order'          => 'DESC',
        );
        
        return get_posts($args);
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook != 'toplevel_page_qa-comments' && $hook != 'post.php') {
            return;
        }
        
        wp_enqueue_script(
            'qa-admin-js',
            get_template_directory_uri() . '/js/qa-admin.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_localize_script('qa-admin-js', 'qaAdminData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('qa_admin_nonce'),
            'importNonce' => wp_create_nonce('qa_import_nonce'),
        ));
        
        // Add inline CSS
        $custom_css = "
            .tab-content { margin-top: 20px; }
            .nav-tab-wrapper { margin-bottom: 20px; }
            .qa-answer-modal {
                display: none;
                position: fixed;
                z-index: 9999;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
                overflow: auto;
                background-color: rgba(0,0,0,0.4);
            }
            .qa-answer-modal-content {
                background-color: #fefefe;
                margin: 10% auto;
                padding: 20px;
                border: 1px solid #888;
                width: 60%;
                border-radius: 5px;
                box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            }
            .qa-answer-modal-close {
                color: #aaa;
                float: right;
                font-size: 28px;
                font-weight: bold;
                cursor: pointer;
            }
            .qa-answer-modal-close:hover {
                color: black;
            }
            .qa-answer-form-group {
                margin-bottom: 15px;
            }
            .qa-answer-form-group label {
                display: block;
                margin-bottom: 5px;
                font-weight: bold;
            }
            .qa-answer-textarea {
                width: 100%;
                min-height: 150px;
                padding: 10px;
                border: 1px solid #ddd;
                border-radius: 4px;
            }
            .qa-answer-submit {
                background-color: #0073aa;
                color: white;
                border: none;
                padding: 10px 15px;
                border-radius: 4px;
                cursor: pointer;
            }
            .qa-answer-submit:hover {
                background-color: #005177;
            }
        ";
        wp_add_inline_style('admin-bar', $custom_css);
    }
    
    /**
     * AJAX handler for approving comments
     */
    public function approve_qa_comment() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qa_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Get comment ID
        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;
        
        if ($comment_id <= 0) {
            wp_send_json_error('Invalid comment ID');
        }
        
        // Update comment status
        $updated = wp_update_post(array(
            'ID'          => $comment_id,
            'post_status' => 'publish',
        ));
        
        if ($updated) {
            wp_send_json_success('Comment approved successfully');
        } else {
            wp_send_json_error('Failed to approve comment');
        }
    }
    
    /**
     * AJAX handler for deleting comments
     */
    public function delete_qa_comment() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qa_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Get comment ID
        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;
        
        if ($comment_id <= 0) {
            wp_send_json_error('Invalid comment ID');
        }
        
        // Delete comment
        $deleted = wp_delete_post($comment_id, true);
        
        if ($deleted) {
            wp_send_json_success('Comment deleted successfully');
        } else {
            wp_send_json_error('Failed to delete comment');
        }
    }
    
    /**
     * AJAX handler for submitting comments from frontend
     */
    public function submit_qa_comment() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qa_submit_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Get form data
        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        $mobile = isset($_POST['mobile']) ? sanitize_text_field($_POST['mobile']) : '';
        $question = isset($_POST['question']) ? sanitize_text_field($_POST['question']) : '';
        $service = isset($_POST['service']) ? intval($_POST['service']) : 0;
        
        // Validate data
        if (empty($name) || empty($question)) {
            wp_send_json_error('لطفا نام و پرسش خود را وارد کنید.');
        }
        
        // Validate email if provided
        if (!empty($email) && !is_email($email)) {
            wp_send_json_error('لطفا یک ایمیل معتبر وارد کنید.');
        }
        
        // Validate mobile if provided
        if (!empty($mobile) && !preg_match('/^09[0-9]{9}$/', $mobile)) {
            wp_send_json_error('لطفا یک شماره موبایل معتبر وارد کنید.');
        }
        
        // Create new comment
        $comment_id = wp_insert_post(array(
            'post_title'   => $question,
            'post_type'    => 'qa_comment',
            'post_status'  => 'pending',
            'post_author'  => 1, // Default to admin
        ));
        
        if ($comment_id) {
            // Add meta data
            update_post_meta($comment_id, '_qa_name', $name);
            update_post_meta($comment_id, '_qa_email', $email);
            update_post_meta($comment_id, '_qa_mobile', $mobile);
            update_post_meta($comment_id, '_qa_service', $service);
            
            wp_send_json_success('پرسش شما با موفقیت ثبت شد و پس از تایید نمایش داده خواهد شد.');
        } else {
            wp_send_json_error('خطا در ثبت پرسش. لطفا دوباره تلاش کنید.');
        }
    }
    
    /**
     * AJAX handler for saving answers
     */
    public function save_qa_answer() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qa_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Get data
        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;
        $answer = isset($_POST['answer']) ? wp_kses_post($_POST['answer']) : '';
        
        if ($comment_id <= 0) {
            wp_send_json_error('Invalid comment ID');
        }
        
        // Save answer
        update_post_meta($comment_id, '_qa_answer', $answer);
        
        wp_send_json_success('Answer saved successfully');
    }

    /**
     * AJAX handler for exporting Q&A data
     */
    public function export_qa_data() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qa_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Get all Q&A comments
        $args = array(
            'post_type'      => 'qa_comment',
            'post_status'    => array('pending', 'publish'),
            'posts_per_page' => -1,
            'orderby'        => 'date',
            'order'          => 'DESC',
        );
        
        $comments = get_posts($args);
        $export_data = array();
        
        foreach ($comments as $comment) {
            $export_data[] = array(
                'id'      => $comment->ID,
                'title'   => $comment->post_title,
                'content' => $comment->post_content,
                'status'  => $comment->post_status,
                'date'    => $comment->post_date,
                'name'    => get_post_meta($comment->ID, '_qa_name', true),
                'email'   => get_post_meta($comment->ID, '_qa_email', true),
                'mobile'  => get_post_meta($comment->ID, '_qa_mobile', true),
                'service' => get_post_meta($comment->ID, '_qa_service', true),
                'answer'  => get_post_meta($comment->ID, '_qa_answer', true),
            );
        }
        
        wp_send_json_success($export_data);
    }
    
    /**
     * AJAX handler for importing Q&A data
     */
    public function import_qa_data() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'qa_import_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Check if file was uploaded
        if (!isset($_FILES['import_file']) || empty($_FILES['import_file']['tmp_name'])) {
            wp_send_json_error('لطفاً یک فایل JSON انتخاب کنید.');
        }
        
        // Get selected service
        $selected_service = isset($_POST['service']) ? intval($_POST['service']) : 0;
        
        $file_content = file_get_contents($_FILES['import_file']['tmp_name']);
        $import_data = json_decode($file_content, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error('فایل JSON نامعتبر است.');
        }
        
        if (!is_array($import_data)) {
            wp_send_json_error('ساختار فایل JSON نامعتبر است.');
        }
        
        $imported_count = 0;
        
        foreach ($import_data as $item) {
            if (!isset($item['title']) || empty($item['title'])) {
                continue;
            }
            
            // Create new comment
            $comment_id = wp_insert_post(array(
                'post_title'   => sanitize_text_field($item['title']),
                'post_content' => isset($item['content']) ? wp_kses_post($item['content']) : '',
                'post_type'    => 'qa_comment',
                'post_status'  => isset($item['status']) && in_array($item['status'], array('pending', 'publish')) ? $item['status'] : 'pending',
                'post_date'    => isset($item['date']) ? $item['date'] : current_time('mysql'),
            ));
            
            if ($comment_id && !is_wp_error($comment_id)) {
                // Add meta data
                update_post_meta($comment_id, '_qa_name', isset($item['name']) ? sanitize_text_field($item['name']) : '');
                update_post_meta($comment_id, '_qa_email', isset($item['email']) ? sanitize_email($item['email']) : '');
                update_post_meta($comment_id, '_qa_mobile', isset($item['mobile']) ? sanitize_text_field($item['mobile']) : '');
                update_post_meta($comment_id, '_qa_service', $selected_service);
                update_post_meta($comment_id, '_qa_answer', isset($item['answer']) ? wp_kses_post($item['answer']) : '');
                
                $imported_count++;
            }
        }
        
        wp_send_json_success(sprintf('%d مورد با موفقیت درون‌ریزی شد.', $imported_count));
    }
}

// Initialize the Q&A system
new Mohtavanegar_QA_System();