<?php
/**
 * Theme Updater Class
 *
 * @package    Stand_Alone_Project
 * @subpackage Theme_Updater
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

class Stand_Alone_Theme_Updater {
    /**
     * Option name for storing site registration data
     *
     * @var string
     */
    private $registration_option = 'stand_alone_theme_registration';
    /**
     * The theme slug
     *
     * @var string
     */
    private $theme_slug;

    /**
     * The current theme version
     *
     * @var string
     */
    private $version;

    /**
     * The update API URL
     *
     * @var string
     */
    private $api_url = 'https://medpress.net/wp-json/dptm-theme-updater/v1';

    /**
     * Initialize the class
     */
    public function __construct() {
        // Get current theme info
        $theme = wp_get_theme();
        $this->theme_slug = $theme->get_template(); // تغییر از get_stylesheet() به get_template()
        $this->version = $theme->get('Version');
        
        // Log theme info for debugging
        error_log('Theme Updater Initialized');
        error_log('Theme Slug: ' . $this->theme_slug);
        error_log('Current Version: ' . $this->version);

        // Set up hooks
        add_filter('pre_set_site_transient_update_themes', array($this, 'check_for_update'));
        add_filter('themes_api', array($this, 'theme_api_call'), 10, 3);
        add_filter('site_transient_update_themes', array($this, 'check_for_update'));
        
        // Register theme activation hook
        add_action('after_switch_theme', array($this, 'register_site_on_activation'));
    }

    /**
     * Check for theme updates
     */
    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        // Get the remote version info
        $remote_info = $this->get_remote_info();
        
        if (!$remote_info || is_wp_error($remote_info)) {
            error_log('Failed to get remote info or invalid response');
            return $transient;
        }

        error_log('Current Version: ' . $this->version);
        error_log('Remote Version: ' . $remote_info['version']);
        error_log('Version Compare: ' . version_compare($this->version, $remote_info['version'], '<'));

        // If a newer version is available, add the update
        if (version_compare($this->version, $remote_info['version'], '<')) {
            $theme_data = wp_get_theme();
            $theme_slug = $theme_data->get_template();
            
            // Ensure the download URL is properly encoded and clean
            $download_url = trim($remote_info['download_url']);
            $download_url = str_replace('&#8230;', '', $download_url); // Remove any ellipsis
            $download_url = esc_url_raw($download_url); // Properly encode the URL
            
            $transient->response[$theme_slug] = array(
                'theme'       => $theme_slug,
                'new_version' => $remote_info['version'],
                'url'         => $this->api_url . '/changelog',
                'package'     => $download_url,
            );
            
            error_log('Download URL set to: ' . $download_url);
            
            error_log('Update available: ' . $remote_info['version']);
        } else {
            error_log('No update available or versions are the same');
            // حذف اطلاعات آپدیت اگر نسخه‌ها یکسان باشند
            $theme_slug = wp_get_theme()->get_template();
            if (isset($transient->response[$theme_slug])) {
                unset($transient->response[$theme_slug]);
            }
        }

        return $transient;
    }
    
    /**
     * Clear update cache
     */
    private function clear_update_cache() {
        delete_site_transient('update_themes');
        wp_clean_themes_cache();
        error_log('Update cache cleared');
    }

    /**
     * Get theme info from remote API
     */
    private function get_remote_info() {
        $request_url = $this->api_url . '/info?slug=' . $this->theme_slug;
        error_log('Fetching remote info from: ' . $request_url);
        
        $request = wp_remote_get($request_url, array(
            'timeout' => 10, // افزایش تایم‌اوت به 10 ثانیه
        ));

        if (is_wp_error($request)) {
            error_log('API Request Error: ' . $request->get_error_message());
            return false;
        }

        $response_code = wp_remote_retrieve_response_code($request);
        $response_body = wp_remote_retrieve_body($request);
        error_log('API Response Code: ' . $response_code);
        error_log('API Response Body: ' . $response_body);

        if ($response_code !== 200) {
            error_log('API Error: Invalid response code');
            return false;
        }

        $response = json_decode($response_body, true);
        
        if (empty($response) || !isset($response['version'])) {
            error_log('API Error: Invalid response format');
            return false;
        }

        error_log('Remote Version: ' . $response['version']);
        error_log('Download URL: ' . ($response['download_url'] ?? 'Not provided'));
        
        return array(
            'version' => $response['version'],
            'download_url' => $response['download_url'] ?? '',
            'changelog' => $response['sections']['changelog'] ?? 'بدون تغییرات ثبت شده'
        );
    }

    /**
     * Theme API call
     */
    public function theme_api_call($false, $action, $args) {
        if (!isset($args->slug) || $args->slug !== $this->theme_slug) {
            return $false;
        }

        // Get the remote info
        $remote_info = $this->get_remote_info();
        
        if (!$remote_info || is_wp_error($remote_info)) {
            return $false;
        }

        // Get current theme data
        $theme_data = wp_get_theme();
        
        // Prepare response
        $response = array(
            'slug'          => $this->theme_slug,
            'name'          => $theme_data->get('Name'),
            'version'       => $remote_info['version'],
            'author'        => $theme_data->get('Author'),
            'download_link' => $remote_info['download_url'],
            'sections'      => array(
                'description' => $theme_data->get('Description'),
                'changelog'   => $remote_info['changelog']
            ),
            'requires'      => '5.0',
            'requires_php'  => '7.4',
        );

        return (object) $response;
    }

    /**
     * Register site on theme activation
     */
    public function register_site_on_activation() {
        // Check if already registered
        $registration = get_option($this->registration_option, array());
        
        if (!empty($registration['site_id']) && !empty($registration['registration_date'])) {
            error_log('Site already registered. Skipping registration.');
            return;
        }
        
        // Prepare registration data
        $site_name = get_bloginfo('name');
        $site_url = home_url('/');
        
        $response = wp_remote_post(
            $this->api_url . '/register-site',
            array(
                'timeout' => 15,
                'headers' => array('Content-Type' => 'application/json; charset=utf-8'),
                'body' => json_encode(array(
                    'site_name' => $site_name,
                    'site_url' => $site_url
                ))
            )
        );
        
        if (is_wp_error($response)) {
            error_log('Registration failed: ' . $response->get_error_message());
            return;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($response_code === 200 && !empty($body['success']) && $body['success'] === true) {
            // Save registration data
            update_option($this->registration_option, array(
                'site_id' => $body['site_id'],
                'site_name' => $body['site_name'],
                'site_url' => $body['site_url'],
                'registration_date' => $body['registration_date'],
                'last_checked' => current_time('mysql')
            ));
            
            error_log('Site registered successfully. Site ID: ' . $body['site_id']);
        } else {
            $error_message = isset($body['message']) ? $body['message'] : 'Unknown error';
            error_log('Registration failed. Status: ' . $response_code . ' - ' . $error_message);
        }
    }
}

// Initialize the updater
new Stand_Alone_Theme_Updater();
