(function($) {
    'use strict';

    // Aparat Playlist Widget JavaScript
    class AparatPlaylistWidget {
        constructor() {
            this.init();
        }

        init() {
            // Wait for DOM to be ready
            $(document).ready(() => {
                this.setupVideoWrappers();
                this.setupResponsiveHandlers();
                this.setupAccessibility();
                this.setupErrorHandling();
            });
        }

        setupVideoWrappers() {
            // Find all video wrappers and setup proper aspect ratios
            $('.aparat-video-wrapper').each(function() {
                const $wrapper = $(this);
                const $iframe = $wrapper.find('iframe');
                
                if ($iframe.length) {
                    // Ensure iframe has proper attributes
                    $iframe.attr({
                        'allowfullscreen': 'true',
                        'allow': 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture'
                    });
                }
            });
        }

        setupResponsiveHandlers() {
            // Handle window resize events
            let resizeTimer;
            $(window).on('resize', () => {
                clearTimeout(resizeTimer);
                resizeTimer = setTimeout(() => {
                    this.handleResize();
                }, 250);
            });

            // Initial resize handling
            this.handleResize();
        }

        handleResize() {
            // Adjust video containers based on viewport
            const viewportWidth = window.innerWidth;
            const $gridContainers = $('.elementor-view-grid .aparat-container');
            
            if (viewportWidth <= 480) {
                // Mobile: single column
                $gridContainers.css('grid-template-columns', '1fr');
            } else if (viewportWidth <= 768) {
                // Tablet: 2 columns max
                $gridContainers.css('grid-template-columns', 'repeat(auto-fit, minmax(250px, 1fr))');
            } else {
                // Desktop: use Elementor's column settings
                $gridContainers.css('grid-template-columns', '');
            }
        }

        setupAccessibility() {
            // Add keyboard navigation for video items
            $('.aparat-item').attr('tabindex', '0');

            $('.aparat-item').on('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    const $iframe = $(this).find('iframe').first();
                    if ($iframe.length) {
                        // Focus on the iframe
                        $iframe.focus();
                    }
                }
            });

            // Add ARIA labels for better accessibility
            $('.aparat-video-wrapper').each(function(index) {
                const $wrapper = $(this);
                const $description = $wrapper.siblings('.aparat-description');
                const description = $description.text().trim();
                
                if (description) {
                    $wrapper.attr('aria-label', `ویدیو ${index + 1}: ${description}`);
                } else {
                    $wrapper.attr('aria-label', `ویدیو ${index + 1}`);
                }
            });
        }

        setupErrorHandling() {
            // Monitor iframe loading
            $('.aparat-video-wrapper iframe').on('load', function() {
                const $wrapper = $(this).parent();
                $wrapper.removeClass('loading');
                
                // Check if iframe loaded successfully
                try {
                    const iframeDoc = this.contentDocument || this.contentWindow.document;
                    if (iframeDoc) {
                        $wrapper.removeClass('error');
                    }
                } catch (e) {
                    // Cross-origin iframe, assume it's loading correctly
                    $wrapper.removeClass('error');
                }
            }).on('error', function() {
                const $wrapper = $(this).parent();
                $wrapper.addClass('error').html(
                    '<div class="error-message">خطا در بارگذاری ویدیو. لطفاً لینک را بررسی کنید.</div>'
                );
            });

            // Add loading state initially
            $('.aparat-video-wrapper').each(function() {
                const $wrapper = $(this);
                if (!$wrapper.find('iframe').length) {
                    $wrapper.addClass('error').html(
                        '<div class="error-message">لینک ویدیو نامعتبر است.</div>'
                    );
                } else {
                    $wrapper.addClass('loading');
                }
            });
        }

        // Utility function to validate Aparat URLs
        static validateAparatUrl(url) {
            if (!url) return false;
            
            const aparatRegex = /^https?:\/\/(www\.)?aparat\.com\/v\/([a-zA-Z0-9_-]+)/;
            return aparatRegex.test(url);
        }

        // Utility function to extract video ID
        static extractVideoId(url) {
            if (!url) return '';
            
            const match = url.match(/^https?:\/\/(www\.)?aparat\.com\/v\/([a-zA-Z0-9_-]+)/);
            return match ? match[2] : '';
        }
    }

    // Initialize the widget
    const aparatPlaylistWidget = new AparatPlaylistWidget();

    // Make it available globally for potential external use
    window.AparatPlaylistWidget = AparatPlaylistWidget;

    // Remove loading spinner after page fully loads
    $(window).on('load', function() {
        $('.aparat-video-wrapper.loading').each(function() {
            this.style.setProperty('content', 'none', 'important');
        });
    });

    // Handle Elementor editor preview
    if (typeof window.elementorFrontend !== 'undefined' && 
        typeof window.elementorFrontend.hooks !== 'undefined' && 
        typeof window.elementorFrontend.hooks.addAction === 'function') {
        elementorFrontend.hooks.addAction('frontend/element_ready/aparat_playlist_widget.default', function($scope) {
            // Re-initialize when widget is loaded in Elementor editor
            new AparatPlaylistWidget();
        });
    }

})(jQuery);

// Additional utility functions for potential external use
window.AparatUtils = {
    // Generate embed code dynamically
    generateEmbedCode: function(videoUrl) {
        const videoId = AparatPlaylistWidget.extractVideoId(videoUrl);
        if (!videoId) return '';
        
        const randomDivId = 'aparat_' + Math.random().toString(36).substr(2, 9);
        return `<div id="${randomDivId}"><script type="text/JavaScript" src="https://www.aparat.com/embed/${videoId}?data[rnddiv]=${randomDivId}&data[responsive]=yes"></script></div>`;
    },

    // Check if video is valid
    isValidVideo: function(videoUrl) {
        return AparatPlaylistWidget.validateAparatUrl(videoUrl);
    },

    // Get video thumbnail URL (if needed in future)
    getThumbnailUrl: function(videoUrl) {
        const videoId = AparatPlaylistWidget.extractVideoId(videoUrl);
        if (!videoId) return '';
        
        // Aparat typically uses this pattern for thumbnails
        return `https://www.aparat.com/videothumb/embed/${videoId}.jpg`;
    }
};
