(function($) {
    'use strict';

    // Cache DOM elements and settings
    const CACHE = {
        widgets: null,
        resizeTimer: null,
        editorTimeout: null
    };

    // Initialize carousels
    function initCarousels() {
        if (!CACHE.widgets) {
            CACHE.widgets = $('.elementor-widget-mohtavanegar_posts_display');
        }

        CACHE.widgets.each(function() {
            const $widget = $(this);
            const $carousel = $widget.find('.mohtavanegar-carousel');
            
            if ($carousel.length === 0) {
                return;
            }
            
            try {
                // Get settings from data attributes
                const settings = {
                    showArrows: $widget.data('show-arrows') === 'yes',
                    showDots: $widget.data('show-dots') === 'yes',
                    autoplay: $widget.data('autoplay') === 'yes' ? {
                        delay: parseInt($widget.data('autoplay-speed')) || 5000,
                        disableOnInteraction: false,
                        pauseOnMouseEnter: $widget.data('pause-on-hover') === 'yes'
                    } : false,
                    loop: $widget.data('infinite') === 'yes',
                    columns: {
                        desktop: parseInt($widget.data('columns')) || 3,
                        tablet: parseInt($widget.data('columns-tablet')) || 2,
                        mobile: parseInt($widget.data('columns-mobile')) || 1
                    }
                };
                
                // Find the swiper container
                const $swiperContainer = $carousel.find('> .swiper');
                if ($swiperContainer.length === 0) {
                    console.warn('Swiper container not found in widget:', $widget);
                    return;
                }
                
                // Destroy existing Swiper instance if it exists
                if (typeof $swiperContainer[0].swiper !== 'undefined') {
                    $swiperContainer[0].swiper.destroy(true, true);
                }
                
                // Initialize Swiper with error handling
                try {
                    const swiper = new Swiper($swiperContainer[0], {
                        rtl: true,
                        navigation: settings.showArrows ? {
                            nextEl: '.swiper-button-next',
                            prevEl: '.swiper-button-prev',
                        } : false,
                        pagination: settings.showDots ? {
                            el: '.swiper-pagination',
                            clickable: true,
                        } : false,
                        autoplay: settings.autoplay,
                        loop: settings.loop,
                        slidesPerView: settings.columns.mobile,
                        spaceBetween: 20,
                        breakpoints: {
                            // Mobile first approach
                            0: {
                                slidesPerView: settings.columns.mobile,
                                spaceBetween: 10
                            },
                            // Tablet
                            768: {
                                slidesPerView: settings.columns.tablet,
                                spaceBetween: 20
                            },
                            // Desktop
                            1024: {
                                slidesPerView: settings.columns.desktop,
                                spaceBetween: 30
                            }
                        },
                        touchEventsTarget: 'container',
                        touchRatio: 1,
                        grabCursor: true,
                        speed: 500,
                        watchOverflow: true,
                        observer: true,
                        observeParents: true,
                        on: {
                            init: function() {
                                $widget.addClass('swiper-initialized');
                                this.update(); // Force update on init
                            },
                            resize: function() {
                                this.update();
                            },
                            error: function(error) {
                                console.error('Swiper error:', error);
                            }
                        }
                    });
                    
                    // Store Swiper instance
                    $carousel.data('swiper', swiper);
                    
                    // Force update on window resize
                    $(window).on('resize', function() {
                        if (swiper) {
                            swiper.update();
                        }
                    });
                    
                } catch (e) {
                    console.error('Error initializing Swiper:', e);
                }
                
            } catch (e) {
                console.error('Error processing widget:', e);
            }
        });
    }

    // Initialize on document ready
    $(document).ready(function() {
        initCarousels();
    });

    // Reinitialize on Elementor editor events with debounce
    if (window.elementor) {
        function debouncedInit() {
            clearTimeout(CACHE.editorTimeout);
            CACHE.editorTimeout = setTimeout(initCarousels, 300);
        }
        
        elementor.hooks.addAction('panel/open_editor/widget', debouncedInit);
        elementor.hooks.addAction('panel/open_editor/widget/mohtavanegar_posts_display', debouncedInit);
        elementor.channels.editor.on('change', debouncedInit);
        
        // Handle elementor preview refresh
        if (elementor.on) {
            elementor.on('preview:loaded', function() {
                setTimeout(initCarousels, 500);
            });
        }
    }

    // Reinitialize on Elementor preview refresh
    if (window.elementorFrontend) {
        $(window).on('elementor/frontend/init', function() {
            setTimeout(initCarousels, 500);
        });
    }

    // Reinitialize on window load
    window.addEventListener('load', function() {
        setTimeout(initCarousels, 1000);
    });

    // Make initCarousels available globally for AJAX calls
    window.initMohtavanegarCarousels = initCarousels;
    
    // Handle AJAX loaded content
    $(document).ajaxComplete(function() {
        setTimeout(initCarousels, 500);
    });

    // Reinit on window resize with debounce
    window.addEventListener('resize', function() {
        clearTimeout(CACHE.resizeTimer);
        CACHE.resizeTimer = setTimeout(initCarousels, 250);
    });

})(jQuery);
