<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Fix menu import issues
 */

/**
 * Fix invalid menu items after import
 */
function mohtavanegar_fix_imported_menus() {
    // Get all menus
    $menus = wp_get_nav_menus();
    
    if (empty($menus)) {
        return;
    }
    
    error_log('Starting menu fix process for ' . count($menus) . ' menus');
    
    foreach ($menus as $menu) {
        $menu_items = wp_get_nav_menu_items($menu->term_id);
        
        if (empty($menu_items)) {
            continue;
        }
        
        error_log('Processing ' . count($menu_items) . ' items for menu: ' . $menu->name);
        
        foreach ($menu_items as $menu_item) {
            $fixed = false;
            
            // Fix custom links
            if ($menu_item->type === 'custom') {
                // Process URL
                $url = $menu_item->url;
                
                // If the URL is invalid or empty, try to fix it
                if (empty($url) || $url === '#' || $url === 'invalid') {
                    // Set a default URL to the homepage
                    $url = home_url('/');
                    $fixed = true;
                } else if (filter_var($url, FILTER_VALIDATE_URL)) {
                    // URL is valid, but might be from the old domain
                    // No need to change it
                } else {
                    // URL might be a relative path without leading slash
                    if (substr($url, 0, 1) !== '/') {
                        $url = '/' . $url;
                        $fixed = true;
                    }
                }
                
                if ($fixed) {
                    // Update the menu item
                    wp_update_nav_menu_item($menu->term_id, $menu_item->ID, array(
                        'menu-item-title' => $menu_item->title,
                        'menu-item-url' => $url,
                        'menu-item-status' => 'publish'
                    ));
                    
                    error_log('Fixed custom menu item: ' . $menu_item->title . ', new URL: ' . $url);
                }
            }
            // Fix post/page/taxonomy links
            else if ($menu_item->type === 'post_type' || $menu_item->type === 'taxonomy') {
                // Check if the object exists
                $object_id = $menu_item->object_id;
                $object_type = $menu_item->object;
                
                if ($menu_item->type === 'post_type') {
                    // For post types, check if the post exists
                    $post = get_post($object_id);
                    
                    if (!$post) {
                        // Post doesn't exist, try to find a post with the same title
                        $posts = get_posts(array(
                            'post_type' => $object_type,
                            'posts_per_page' => 1,
                            'title' => $menu_item->title,
                            'post_status' => 'publish'
                        ));
                        
                        if (!empty($posts)) {
                            // Found a post with the same title, update the menu item
                            wp_update_nav_menu_item($menu->term_id, $menu_item->ID, array(
                                'menu-item-title' => $menu_item->title,
                                'menu-item-object-id' => $posts[0]->ID,
                                'menu-item-object' => $object_type,
                                'menu-item-type' => 'post_type',
                                'menu-item-status' => 'publish'
                            ));
                            
                            error_log('Fixed post_type menu item: ' . $menu_item->title . ', new object ID: ' . $posts[0]->ID);
                        } else {
                            // No post found, convert to custom link
                            wp_update_nav_menu_item($menu->term_id, $menu_item->ID, array(
                                'menu-item-title' => $menu_item->title,
                                'menu-item-url' => home_url('/'),
                                'menu-item-status' => 'publish',
                                'menu-item-type' => 'custom'
                            ));
                            
                            error_log('Converted invalid post_type menu item to custom link: ' . $menu_item->title);
                        }
                    }
                } else if ($menu_item->type === 'taxonomy') {
                    // For taxonomies, check if the term exists
                    $term = get_term($object_id, $object_type);
                    
                    if (!$term || is_wp_error($term)) {
                        // Term doesn't exist, try to find a term with the same name
                        $terms = get_terms(array(
                            'taxonomy' => $object_type,
                            'name' => $menu_item->title,
                            'hide_empty' => false,
                            'number' => 1
                        ));
                        
                        if (!empty($terms) && !is_wp_error($terms)) {
                            // Found a term with the same name, update the menu item
                            wp_update_nav_menu_item($menu->term_id, $menu_item->ID, array(
                                'menu-item-title' => $menu_item->title,
                                'menu-item-object-id' => $terms[0]->term_id,
                                'menu-item-object' => $object_type,
                                'menu-item-type' => 'taxonomy',
                                'menu-item-status' => 'publish'
                            ));
                            
                            error_log('Fixed taxonomy menu item: ' . $menu_item->title . ', new object ID: ' . $terms[0]->term_id);
                        } else {
                            // No term found, convert to custom link
                            wp_update_nav_menu_item($menu->term_id, $menu_item->ID, array(
                                'menu-item-title' => $menu_item->title,
                                'menu-item-url' => home_url('/'),
                                'menu-item-status' => 'publish',
                                'menu-item-type' => 'custom'
                            ));
                            
                            error_log('Converted invalid taxonomy menu item to custom link: ' . $menu_item->title);
                        }
                    }
                }
            }
        }
    }
    
    error_log('Menu fix process completed');
}

// Add a hook to run the fix after import
add_action('mohtavanegar_after_import', 'mohtavanegar_fix_imported_menus');
