<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Menus Import/Export Functions
 */

/**
 * Export menus
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_menus($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create menus directory
    $menus_dir = $export_dir . '/menus';
    if (!is_dir($menus_dir)) {
        mkdir($menus_dir, 0755, true);
    }
    
    // Get all menu locations
    $menu_locations = get_nav_menu_locations();
    
    // Get all menus
    $menus = wp_get_nav_menus();
    
    $menus_data = array();
    $menu_items_data = array();
    
    foreach ($menus as $menu) {
        // Find menu location
        $locations = array();
        foreach ($menu_locations as $location => $menu_id) {
            if ($menu_id === $menu->term_id) {
                $locations[] = $location;
            }
        }
        
        $menus_data[] = array(
            'term_id' => $menu->term_id,
            'name' => $menu->name,
            'slug' => $menu->slug,
            'locations' => $locations
        );
        
        // Get menu items
        $menu_items = wp_get_nav_menu_items($menu->term_id);
        
        if ($menu_items) {
            foreach ($menu_items as $item) {
                $menu_items_data[$menu->term_id][] = array(
                    'ID' => $item->ID,
                    'db_id' => $item->db_id,
                    'menu_item_parent' => $item->menu_item_parent,
                    'object_id' => $item->object_id,
                    'object' => $item->object,
                    'type' => $item->type,
                    'type_label' => $item->type_label,
                    'title' => $item->title,
                    'url' => $item->url,
                    'target' => $item->target,
                    'attr_title' => $item->attr_title,
                    'description' => $item->description,
                    'classes' => $item->classes,
                    'xfn' => $item->xfn,
                    'menu_order' => $item->menu_order,
                    'original_title' => get_post_meta($item->ID, '_menu_item_original_title', true),
                    'meta' => array(
                        '_menu_item_type' => get_post_meta($item->ID, '_menu_item_type', true),
                        '_menu_item_menu_item_parent' => get_post_meta($item->ID, '_menu_item_menu_item_parent', true),
                        '_menu_item_object_id' => get_post_meta($item->ID, '_menu_item_object_id', true),
                        '_menu_item_object' => get_post_meta($item->ID, '_menu_item_object', true),
                        '_menu_item_target' => get_post_meta($item->ID, '_menu_item_target', true),
                        '_menu_item_classes' => get_post_meta($item->ID, '_menu_item_classes', true),
                        '_menu_item_xfn' => get_post_meta($item->ID, '_menu_item_xfn', true),
                        '_menu_item_url' => get_post_meta($item->ID, '_menu_item_url', true),
                    )
                );
            }
        }
    }
    
    // Save menus data
    $menus_file = $menus_dir . '/menus.json';
    file_put_contents($menus_file, json_encode($menus_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Save menu items data
    $menu_items_file = $menus_dir . '/menu_items.json';
    file_put_contents($menu_items_file, json_encode($menu_items_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($menus);
    $result['message'] = sprintf(__('%d منو با موفقیت برون‌ریزی شدند.', 'mohtavanegar'), count($menus));
    
    return $result;
}

/**
 * Import menus
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_menus($import_dir) {
    // Add debugging
    error_log('Starting menu import process');
    
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    $menus_dir = $import_dir . '/menus';
    
    // Check if menus directory exists
    if (!is_dir($menus_dir)) {
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری منوها یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // Import menus
    $menus_file = $menus_dir . '/menus.json';
    $menu_items_file = $menus_dir . '/menu_items.json';
    
    if (!file_exists($menus_file) || !file_exists($menu_items_file)) {
        $result['status'] = 'error';
        $result['message'] = __('فایل‌های منو یافت نشدند.', 'mohtavanegar');
        return $result;
    }
    
    $menus_data = json_decode(file_get_contents($menus_file), true);
    $menu_items_data = json_decode(file_get_contents($menu_items_file), true);
    
    if (!$menus_data) {
        $result['status'] = 'error';
        $result['message'] = __('داده‌های منو نامعتبر هستند.', 'mohtavanegar');
        return $result;
    }
    
    // Create/update menus
    $menu_id_mapping = array();
    $imported_count = 0;
    
    foreach ($menus_data as $menu_data) {
        // Check if menu exists
        $existing_menu = wp_get_nav_menu_object($menu_data['slug']);
        
        if ($existing_menu) {
            // Update existing menu
            $menu_id = $existing_menu->term_id;
            wp_update_nav_menu_object($menu_id, array('menu-name' => $menu_data['name']));
        } else {
            // Create new menu
            $menu_id = wp_create_nav_menu($menu_data['name']);
        }
        
        if (!is_wp_error($menu_id)) {
            $menu_id_mapping[$menu_data['term_id']] = $menu_id;
            
            // Set menu locations
            if (!empty($menu_data['locations'])) {
                $locations = get_theme_mod('nav_menu_locations');
                
                foreach ($menu_data['locations'] as $location) {
                    $locations[$location] = $menu_id;
                }
                
                set_theme_mod('nav_menu_locations', $locations);
            }
            
            $imported_count++;
        }
    }
    
    // Get post ID mappings to handle post references correctly
    $post_id_mapping = array();
    
    // Map post slugs to IDs for posts
    $posts = get_posts(array('post_type' => 'post', 'posts_per_page' => -1, 'post_status' => 'publish'));
    foreach ($posts as $post) {
        $post_id_mapping['post_' . $post->post_name] = $post->ID;
    }
    
    // Map post slugs to IDs for pages
    $pages = get_posts(array('post_type' => 'page', 'posts_per_page' => -1, 'post_status' => 'publish'));
    foreach ($pages as $page) {
        $post_id_mapping['page_' . $page->post_name] = $page->ID;
    }
    
    // Map post slugs to IDs for custom post types
    $custom_post_types = get_post_types(array('_builtin' => false), 'names');
    foreach ($custom_post_types as $post_type) {
        $custom_posts = get_posts(array('post_type' => $post_type, 'posts_per_page' => -1, 'post_status' => 'publish'));
        foreach ($custom_posts as $custom_post) {
            $post_id_mapping[$post_type . '_' . $custom_post->post_name] = $custom_post->ID;
        }
    }
    
    error_log('Post ID mapping: ' . print_r($post_id_mapping, true));
    
    // Import menu items
    if (!empty($menu_id_mapping) && !empty($menu_items_data)) {
        foreach ($menu_items_data as $old_menu_id => $items) {
            if (!isset($menu_id_mapping[$old_menu_id])) {
                error_log('Menu ID not found in mapping: ' . $old_menu_id);
                continue;
            }
            
            $new_menu_id = $menu_id_mapping[$old_menu_id];
            error_log('Processing menu items for menu ID: ' . $new_menu_id);
            
            // First, remove existing menu items
            $existing_items = wp_get_nav_menu_items($new_menu_id);
            if ($existing_items) {
                foreach ($existing_items as $item) {
                    wp_delete_post($item->ID, true);
                }
            }
            
            // Create parent menu items first
            $item_id_mapping = array();
            
            // First pass: Add items with no parent
            foreach ($items as $item) {
                if ($item['menu_item_parent'] == 0) {
                    // Process URL for custom links to handle domain changes
                    $url = '';
                    if ($item['type'] === 'custom') {
                        $url = $item['url'];
                        
                        // Handle domain changes by converting to relative URLs when appropriate
                        if (filter_var($url, FILTER_VALIDATE_URL)) {
                            $url_parts = parse_url($url);
                            
                            // If it's an internal link (to the original site), convert to a relative URL
                            if (isset($url_parts['path'])) {
                                // For internal links, just use the path
                                if (strpos($url, get_option('siteurl')) === 0 || 
                                    strpos($url, home_url()) === 0) {
                                    $url = $url_parts['path'];
                                    if (isset($url_parts['query'])) {
                                        $url .= '?' . $url_parts['query'];
                                    }
                                    if (isset($url_parts['fragment'])) {
                                        $url .= '#' . $url_parts['fragment'];
                                    }
                                }
                            }
                        }
                    }
                    
                    // Map object ID to the new ID if it's a post, page, or custom post type
                    $object_id = $item['meta']['_menu_item_object_id'];
                    $object_type = $item['meta']['_menu_item_object'];
                    
                    // Try to find the corresponding post by slug
                    if ($object_type === 'post' || $object_type === 'page' || in_array($object_type, get_post_types(array('_builtin' => false), 'names'))) {
                        // Get the original post to find its slug
                        $original_post_slug = '';
                        
                        // Look up in our mapping by object type and ID
                        $mapping_key = $object_type . '_' . $object_id;
                        if (isset($post_id_mapping[$mapping_key])) {
                            $object_id = $post_id_mapping[$mapping_key];
                        } else {
                            // Try to find by post name/slug
                            $post_by_slug = get_posts(array(
                                'name' => $object_id,
                                'post_type' => $object_type,
                                'posts_per_page' => 1,
                                'post_status' => 'publish'
                            ));
                            
                            if (!empty($post_by_slug)) {
                                $object_id = $post_by_slug[0]->ID;
                            } else {
                                // If we can't find it, set to 0 to avoid invalid references
                                $object_id = 0;
                            }
                        }
                    }
                    
                    error_log('Menu item: ' . $item['title'] . ', Type: ' . $item['meta']['_menu_item_type'] . ', Object: ' . $item['meta']['_menu_item_object'] . ', Object ID: ' . $object_id);
                    
                    $item_args = array(
                        'menu-item-title' => $item['title'],
                        'menu-item-url' => $url,
                        'menu-item-status' => 'publish',
                        'menu-item-type' => $item['meta']['_menu_item_type'],
                        'menu-item-object' => $item['meta']['_menu_item_object'],
                        'menu-item-object-id' => $object_id,
                        'menu-item-target' => $item['meta']['_menu_item_target'],
                        'menu-item-attr-title' => $item['attr_title'],
                        'menu-item-classes' => implode(' ', (array)$item['meta']['_menu_item_classes']),
                        'menu-item-xfn' => $item['meta']['_menu_item_xfn'],
                        'menu-item-description' => $item['description'],
                        'menu-item-position' => $item['menu_order']
                    );
                    
                    $new_item_id = wp_update_nav_menu_item($new_menu_id, 0, $item_args);
                    
                    if (!is_wp_error($new_item_id)) {
                        $item_id_mapping[$item['ID']] = $new_item_id;
                    }
                }
            }
            
            // Second pass: Add items with parents
            foreach ($items as $item) {
                if ($item['menu_item_parent'] != 0) {
                    $parent_id = isset($item_id_mapping[$item['menu_item_parent']]) ? $item_id_mapping[$item['menu_item_parent']] : 0;
                    
                    // Process URL for custom links to handle domain changes
                    $url = '';
                    if ($item['type'] === 'custom') {
                        $url = $item['url'];
                        
                        // Handle domain changes by converting to relative URLs when appropriate
                        if (filter_var($url, FILTER_VALIDATE_URL)) {
                            $url_parts = parse_url($url);
                            
                            // If it's an internal link (to the original site), convert to a relative URL
                            if (isset($url_parts['path'])) {
                                // For internal links, just use the path
                                if (strpos($url, get_option('siteurl')) === 0 || 
                                    strpos($url, home_url()) === 0) {
                                    $url = $url_parts['path'];
                                    if (isset($url_parts['query'])) {
                                        $url .= '?' . $url_parts['query'];
                                    }
                                    if (isset($url_parts['fragment'])) {
                                        $url .= '#' . $url_parts['fragment'];
                                    }
                                }
                            }
                        }
                    }
                    
                    // Map object ID to the new ID if it's a post, page, or custom post type
                    $object_id = $item['meta']['_menu_item_object_id'];
                    $object_type = $item['meta']['_menu_item_object'];
                    
                    // Try to find the corresponding post by slug
                    if ($object_type === 'post' || $object_type === 'page' || in_array($object_type, get_post_types(array('_builtin' => false), 'names'))) {
                        // Get the original post to find its slug
                        $original_post_slug = '';
                        
                        // Look up in our mapping by object type and ID
                        $mapping_key = $object_type . '_' . $object_id;
                        if (isset($post_id_mapping[$mapping_key])) {
                            $object_id = $post_id_mapping[$mapping_key];
                        } else {
                            // Try to find by post name/slug
                            $post_by_slug = get_posts(array(
                                'name' => $object_id,
                                'post_type' => $object_type,
                                'posts_per_page' => 1,
                                'post_status' => 'publish'
                            ));
                            
                            if (!empty($post_by_slug)) {
                                $object_id = $post_by_slug[0]->ID;
                            } else {
                                // If we can't find it, set to 0 to avoid invalid references
                                $object_id = 0;
                            }
                        }
                    }
                    
                    error_log('Menu item: ' . $item['title'] . ', Type: ' . $item['meta']['_menu_item_type'] . ', Object: ' . $item['meta']['_menu_item_object'] . ', Object ID: ' . $object_id);
                    
                    $item_args = array(
                        'menu-item-title' => $item['title'],
                        'menu-item-url' => $url,
                        'menu-item-status' => 'publish',
                        'menu-item-type' => $item['meta']['_menu_item_type'],
                        'menu-item-object' => $item['meta']['_menu_item_object'],
                        'menu-item-object-id' => $object_id,
                        'menu-item-parent-id' => $parent_id,
                        'menu-item-target' => $item['meta']['_menu_item_target'],
                        'menu-item-attr-title' => $item['attr_title'],
                        'menu-item-classes' => implode(' ', (array)$item['meta']['_menu_item_classes']),
                        'menu-item-xfn' => $item['meta']['_menu_item_xfn'],
                        'menu-item-description' => $item['description'],
                        'menu-item-position' => $item['menu_order']
                    );
                    
                    $new_item_id = wp_update_nav_menu_item($new_menu_id, 0, $item_args);
                    
                    if (!is_wp_error($new_item_id)) {
                        $item_id_mapping[$item['ID']] = $new_item_id;
                    }
                }
            }
        }
    }
    
    $result['count'] = $imported_count;
    $result['message'] = sprintf(__('%d منو با موفقیت درون‌ریزی شدند.', 'mohtavanegar'), $imported_count);
    
    return $result;
}
