<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Posts and Categories Import/Export Functions
 */

/**
 * Export posts and categories
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_posts($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create posts directory
    $posts_dir = $export_dir . '/posts';
    if (!is_dir($posts_dir)) {
        mkdir($posts_dir, 0755, true);
    }
    
    // Export categories
    $categories = get_categories(array(
        'hide_empty' => false,
        'orderby' => 'name',
        'order' => 'ASC'
    ));
    
    $categories_data = array();
    foreach ($categories as $category) {
        $categories_data[] = array(
            'term_id' => $category->term_id,
            'name' => $category->name,
            'slug' => $category->slug,
            'description' => $category->description,
            'parent' => $category->parent
        );
    }
    
    $categories_file = $posts_dir . '/categories.json';
    file_put_contents($categories_file, json_encode($categories_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Export posts
    $posts = get_posts(array(
        'post_type' => 'post',
        'post_status' => 'publish',
        'posts_per_page' => -1
    ));
    
    $posts_data = array();
    foreach ($posts as $post) {
        // Get post categories
        $post_categories = wp_get_post_categories($post->ID, array('fields' => 'ids'));
        
        // Get post tags
        $post_tags = wp_get_post_tags($post->ID, array('fields' => 'names'));
        
        // Get post meta
        $post_meta = get_post_meta($post->ID);
        $filtered_meta = array();
        
        // Filter out unnecessary meta
        foreach ($post_meta as $meta_key => $meta_value) {
            // Skip internal meta keys
            if (substr($meta_key, 0, 1) === '_' && $meta_key !== '_thumbnail_id') {
                continue;
            }
            
            $filtered_meta[$meta_key] = maybe_unserialize($meta_value[0]);
        }
        
        // Get featured image
        $featured_image_id = get_post_thumbnail_id($post->ID);
        $featured_image_url = '';
        
        if ($featured_image_id) {
            $featured_image_url = wp_get_attachment_url($featured_image_id);
        }
        
        $posts_data[] = array(
            'ID' => $post->ID,
            'post_title' => $post->post_title,
            'post_content' => $post->post_content,
            'post_excerpt' => $post->post_excerpt,
            'post_date' => $post->post_date,
            'post_name' => $post->post_name,
            'post_status' => $post->post_status,
            'post_author' => $post->post_author,
            'categories' => $post_categories,
            'tags' => $post_tags,
            'meta' => $filtered_meta,
            'featured_image' => $featured_image_url
        );
    }
    
    $posts_file = $posts_dir . '/posts.json';
    file_put_contents($posts_file, json_encode($posts_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($posts);
    $result['message'] = sprintf(__('%d نوشته و %d دسته‌بندی با موفقیت برون‌ریزی شدند.', 'mohtavanegar'), count($posts), count($categories));
    
    return $result;
}

/**
 * Import posts and categories
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_posts($import_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    $posts_dir = $import_dir . '/posts';
    
    // Check if posts directory exists
    if (!is_dir($posts_dir)) {
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری نوشته‌ها یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // Import categories
    $categories_file = $posts_dir . '/categories.json';
    if (file_exists($categories_file)) {
        $categories_data = json_decode(file_get_contents($categories_file), true);
        
        if ($categories_data) {
            // First pass: Create all categories without parents
            $term_id_mapping = array();
            
            foreach ($categories_data as $category) {
                // Check if category exists by slug
                $existing_term = get_term_by('slug', $category['slug'], 'category');
                
                if ($existing_term) {
                    $term_id_mapping[$category['term_id']] = $existing_term->term_id;
                } else {
                    // Create new category
                    $new_term = wp_insert_term(
                        $category['name'],
                        'category',
                        array(
                            'description' => $category['description'],
                            'slug' => $category['slug']
                        )
                    );
                    
                    if (!is_wp_error($new_term)) {
                        $term_id_mapping[$category['term_id']] = $new_term['term_id'];
                    }
                }
            }
            
            // Second pass: Update parent relationships
            foreach ($categories_data as $category) {
                if ($category['parent'] > 0 && isset($term_id_mapping[$category['parent']])) {
                    wp_update_term(
                        $term_id_mapping[$category['term_id']],
                        'category',
                        array(
                            'parent' => $term_id_mapping[$category['parent']]
                        )
                    );
                }
            }
        }
    }
    
    // Import posts
    $posts_file = $posts_dir . '/posts.json';
    if (file_exists($posts_file)) {
        $posts_data = json_decode(file_get_contents($posts_file), true);
        
        if ($posts_data) {
            $imported_count = 0;
            
            foreach ($posts_data as $post_data) {
                // Check if post exists by slug
                $existing_post = get_page_by_path($post_data['post_name'], OBJECT, 'post');
                
                if ($existing_post) {
                    // Update existing post
                    $post_id = $existing_post->ID;
                    
                    wp_update_post(array(
                        'ID' => $post_id,
                        'post_title' => $post_data['post_title'],
                        'post_content' => $post_data['post_content'],
                        'post_excerpt' => $post_data['post_excerpt'],
                        'post_status' => $post_data['post_status']
                    ));
                } else {
                    // Create new post
                    $post_id = wp_insert_post(array(
                        'post_title' => $post_data['post_title'],
                        'post_content' => $post_data['post_content'],
                        'post_excerpt' => $post_data['post_excerpt'],
                        'post_date' => $post_data['post_date'],
                        'post_name' => $post_data['post_name'],
                        'post_status' => $post_data['post_status'],
                        'post_author' => get_current_user_id(),
                        'post_type' => 'post'
                    ));
                }
                
                if ($post_id && !is_wp_error($post_id)) {
                    // Set categories
                    if (!empty($post_data['categories'])) {
                        $new_categories = array();
                        
                        foreach ($post_data['categories'] as $old_category_id) {
                            if (isset($term_id_mapping[$old_category_id])) {
                                $new_categories[] = $term_id_mapping[$old_category_id];
                            }
                        }
                        
                        if (!empty($new_categories)) {
                            wp_set_post_categories($post_id, $new_categories);
                        }
                    }
                    
                    // Set tags
                    if (!empty($post_data['tags'])) {
                        wp_set_post_tags($post_id, $post_data['tags']);
                    }
                    
                    // Set meta
                    if (!empty($post_data['meta'])) {
                        foreach ($post_data['meta'] as $meta_key => $meta_value) {
                            // Special handling for Elementor data
                            if ($meta_key === '_elementor_data' && !empty($meta_value)) {
                                $elementor_data = json_decode($meta_value, true);
                                if ($elementor_data) {
                                    // Get template ID mapping from the elementor templates import
                                    $template_id_mapping = get_option('mohtavanegar_template_id_mapping', array());
                                    
                                    // Process the Elementor data using our processor function
                                    $elementor_data = mohtavanegar_process_elementor_content_data($elementor_data, $template_id_mapping);
                                    
                                    // Update the meta with processed data
                                    update_post_meta($post_id, $meta_key, json_encode($elementor_data));
                                    
                                    // Log the processing
                                    error_log('Processed Elementor data for post ID: ' . $post_id);
                                } else {
                                    update_post_meta($post_id, $meta_key, $meta_value);
                                }
                            } else {
                                update_post_meta($post_id, $meta_key, $meta_value);
                            }
                        }
                    }
                    
                    // Set featured image
                    if (!empty($post_data['featured_image'])) {
                        // The media importer should have already imported this image
                        // We need to find the new attachment ID by the original URL
                        $attachment_id = mohtavanegar_get_attachment_id_by_original_url($post_data['featured_image']);
                        
                        if ($attachment_id) {
                            set_post_thumbnail($post_id, $attachment_id);
                        }
                    }
                    
                    $imported_count++;
                }
            }
            
            $result['count'] = $imported_count;
            $result['message'] = sprintf(__('%d نوشته با موفقیت درون‌ریزی شدند.', 'mohtavanegar'), $imported_count);
        }
    }
    
    return $result;
}

/**
 * Helper function to get attachment ID by original URL
 * Used during import to link featured images
 *
 * @param string $original_url Original URL from export
 * @return int|false Attachment ID or false if not found
 */
function mohtavanegar_get_attachment_id_by_original_url($original_url) {
    global $wpdb;
    
    // Extract the filename from the URL
    $filename = basename(parse_url($original_url, PHP_URL_PATH));
    
    // Search for attachments with this filename
    $attachment = $wpdb->get_col($wpdb->prepare(
        "SELECT ID FROM $wpdb->posts WHERE post_type = 'attachment' AND guid LIKE %s",
        '%' . $filename
    ));
    
    if (!empty($attachment[0])) {
        return $attachment[0];
    }
    
    // Try to find by _mohtavanegar_original_url meta
    $attachment = $wpdb->get_col($wpdb->prepare(
        "SELECT post_id FROM $wpdb->postmeta WHERE meta_key = '_mohtavanegar_original_url' AND meta_value = %s",
        $original_url
    ));
    
    if (!empty($attachment[0])) {
        return $attachment[0];
    }
    
    return false;
}
