<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Services and Categories Import/Export Functions
 */

/**
 * Export services and categories
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_services($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create services directory
    $services_dir = $export_dir . '/services';
    if (!is_dir($services_dir)) {
        mkdir($services_dir, 0755, true);
    }
    
    // Export service categories
    $service_categories = get_terms(array(
        'taxonomy' => 'services_category',
        'hide_empty' => false,
        'orderby' => 'name',
        'order' => 'ASC'
    ));
    
    $categories_data = array();
    foreach ($service_categories as $category) {
        // Get category meta (icon, etc.)
        $category_meta = array();
        $icon = get_term_meta($category->term_id, 'taxonomy_icon', true);
        if ($icon) {
            $category_meta['taxonomy_icon'] = $icon;
            
            // Also get the icon image URL for export
            $icon_image = wp_get_attachment_image_src($icon, 'full');
            if ($icon_image) {
                $category_meta['icon_url'] = $icon_image[0];
            }
        }
        
        $categories_data[] = array(
            'term_id' => $category->term_id,
            'name' => $category->name,
            'slug' => $category->slug,
            'description' => $category->description,
            'parent' => $category->parent,
            'meta' => $category_meta
        );
    }
    
    $categories_file = $services_dir . '/categories.json';
    file_put_contents($categories_file, json_encode($categories_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Export services
    $services = get_posts(array(
        'post_type' => 'services',
        'post_status' => 'publish',
        'posts_per_page' => -1
    ));
    
    $services_data = array();
    foreach ($services as $service) {
        // Get service categories
        $service_categories = wp_get_post_terms($service->ID, 'services_category', array('fields' => 'ids'));
        
        // Get service meta
        $service_meta = get_post_meta($service->ID);
        $filtered_meta = array();
        
        // Filter out unnecessary meta
        foreach ($service_meta as $meta_key => $meta_value) {
            // Skip internal meta keys except for specific ones we want to keep
            if (substr($meta_key, 0, 1) === '_' && 
                !in_array($meta_key, array('_thumbnail_id', '_service_icon', '_service_price', '_service_duration'))) {
                continue;
            }
            
            $filtered_meta[$meta_key] = maybe_unserialize($meta_value[0]);
        }
        
        // Get featured image and its URL
        $featured_image_id = get_post_thumbnail_id($service->ID);
        $featured_image_url = '';
        $featured_image_meta = array();
        
        if ($featured_image_id) {
            $featured_image_url = wp_get_attachment_url($featured_image_id);
            $featured_image_meta['id'] = $featured_image_id;
            $featured_image_meta['url'] = $featured_image_url;
            $featured_image_meta['sizes'] = wp_get_attachment_image_srcset($featured_image_id);
        }
        
        $services_data[] = array(
            'ID' => $service->ID,
            'post_title' => $service->post_title,
            'post_content' => $service->post_content,
            'post_excerpt' => $service->post_excerpt,
            'post_date' => $service->post_date,
            'post_name' => $service->post_name,
            'post_status' => $service->post_status,
            'post_author' => $service->post_author,
            'categories' => $service_categories,
            'meta' => $filtered_meta,
            'featured_image' => $featured_image_url
        );
    }
    
    $services_file = $services_dir . '/services.json';
    file_put_contents($services_file, json_encode($services_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($services);
    $result['message'] = sprintf(__('%d خدمت و %d دسته‌بندی با موفقیت برون‌ریزی شدند.', 'mohtavanegar'), count($services), count($service_categories));
    
    return $result;
}

/**
 * Import services and categories
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_services($import_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    $services_dir = $import_dir . '/services';
    
    // Check if services directory exists
    if (!is_dir($services_dir)) {
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری خدمات یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // Import service categories
    $categories_file = $services_dir . '/categories.json';
    if (file_exists($categories_file)) {
        $categories_data = json_decode(file_get_contents($categories_file), true);
        
        if ($categories_data) {
            $imported_categories = array();
            
            foreach ($categories_data as $category_data) {
                // Check if category already exists
                $existing_category = get_term_by('slug', $category_data['slug'], 'services_category');
                
                if (!$existing_category) {
                    $category = wp_insert_term(
                        $category_data['name'],
                        'services_category',
                        array(
                            'description' => $category_data['description'],
                            'slug' => $category_data['slug'],
                            'parent' => $category_data['parent']
                        )
                    );
                    
                    if (!is_wp_error($category)) {
                        // Import category meta including icon
                        if (!empty($category_data['meta'])) {
                            foreach ($category_data['meta'] as $meta_key => $meta_value) {
                                add_term_meta($category['term_id'], $meta_key, $meta_value);
                            }
                            
                            // If we have icon URL, download and import it
                            if (!empty($category_data['meta']['icon_url'])) {
                                $icon_url = $category_data['meta']['icon_url'];
                                $icon_id = media_sideload_image($icon_url, 0, '', 'id');
                                if (!is_wp_error($icon_id)) {
                                    update_term_meta($category['term_id'], 'taxonomy_icon', $icon_id);
                                }
                            }
                        }
                        $imported_categories[] = $category['term_id'];
                        $term_id_mapping[$category_data['term_id']] = $category['term_id'];
                    } else {
                        $result['status'] = 'error';
                        $result['message'] .= 'Error importing category: ' . $category_data['name'] . '\n';
                    }
                } else {
                    $imported_categories[] = $existing_category->term_id;
                    $term_id_mapping[$category_data['term_id']] = $existing_category->term_id;
                }
            }
            
            // Second pass: Update parent relationships
            foreach ($categories_data as $category) {
                if ($category['parent'] > 0 && isset($term_id_mapping[$category['parent']]) && isset($term_id_mapping[$category['term_id']])) {
                    wp_update_term(
                        $term_id_mapping[$category['term_id']],
                        'services_category',
                        array(
                            'parent' => $term_id_mapping[$category['parent']]
                        )
                    );
                }
            }
        }
    }
    
    // Import services
    $services_file = $services_dir . '/services.json';
    if (file_exists($services_file)) {
        $services_data = json_decode(file_get_contents($services_file), true);
        
        if ($services_data) {
            $imported_count = 0;
            
            foreach ($services_data as $service_data) {
                // Check if service exists by slug
                $existing_service = get_page_by_path($service_data['post_name'], OBJECT, 'services');
                
                if ($existing_service) {
                    // Update existing service
                    $service_id = $existing_service->ID;
                    
                    wp_update_post(array(
                        'ID' => $service_id,
                        'post_title' => $service_data['post_title'],
                        'post_content' => $service_data['post_content'],
                        'post_excerpt' => $service_data['post_excerpt'],
                        'post_status' => $service_data['post_status']
                    ));
                } else {
                    // Create new service
                    $service_id = wp_insert_post(array(
                        'post_title' => $service_data['post_title'],
                        'post_content' => $service_data['post_content'],
                        'post_excerpt' => $service_data['post_excerpt'],
                        'post_date' => $service_data['post_date'],
                        'post_name' => $service_data['post_name'],
                        'post_status' => $service_data['post_status'],
                        'post_author' => get_current_user_id(),
                        'post_type' => 'services'
                    ));
                }
                
                if ($service_id && !is_wp_error($service_id)) {
                    // Set service categories
                    if (!empty($service_data['categories'])) {
                        $new_categories = array();
                        
                        foreach ($service_data['categories'] as $old_category_id) {
                            if (isset($term_id_mapping[$old_category_id])) {
                                $new_categories[] = $term_id_mapping[$old_category_id];
                            }
                        }
                        
                        if (!empty($new_categories)) {
                            wp_set_object_terms($service_id, $new_categories, 'services_category');
                        }
                    }
                    
                    // Set meta
                    if (!empty($service_data['meta'])) {
                        foreach ($service_data['meta'] as $meta_key => $meta_value) {
                            update_post_meta($service_id, $meta_key, $meta_value);
                        }
                    }
                    
                    // Set featured image
                    if (!empty($service_data['featured_image'])) {
                        // The media importer should have already imported this image
                        // We need to find the new attachment ID by the original URL
                        $attachment_id = mohtavanegar_get_attachment_id_by_original_url($service_data['featured_image']);
                        
                        if ($attachment_id) {
                            set_post_thumbnail($service_id, $attachment_id);
                        }
                    }
                    
                    $imported_count++;
                }
            }
            
            $result['count'] = $imported_count;
            $result['message'] = sprintf(__('%d خدمت با موفقیت درون‌ریزی شدند.', 'mohtavanegar'), $imported_count);
        }
    }
    
    return $result;
}