<?php
if (!defined('ABSPATH')) {
    exit;
}

class Mohtavanegar_Reviews_System {
    public function __construct() {
        add_action('init', [$this, 'register_reviews_post_type']);
        add_action('admin_menu', [$this, 'add_reviews_admin_menu']);

        add_action('wp_ajax_submit_patient_review', [$this, 'submit_patient_review']);
        add_action('wp_ajax_nopriv_submit_patient_review', [$this, 'submit_patient_review']);

        add_action('wp_ajax_approve_patient_review', [$this, 'approve_patient_review']);
        add_action('wp_ajax_delete_patient_review', [$this, 'delete_patient_review']);

        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
    }

    private function get_pending_reviews_count() {
        $count = wp_count_posts('patient_review');
        return isset($count->pending) ? (int) $count->pending : 0;
    }

    public function register_reviews_post_type() {
        $labels = [
            'name' => esc_html__('نظر بیماران', 'mohtavanegar'),
            'singular_name' => esc_html__('نظر بیمار', 'mohtavanegar'),
            'menu_name' => esc_html__('نظر بیماران', 'mohtavanegar'),
            'add_new' => esc_html__('افزودن جدید', 'mohtavanegar'),
            'add_new_item' => esc_html__('افزودن نظر جدید', 'mohtavanegar'),
            'edit_item' => esc_html__('ویرایش نظر', 'mohtavanegar'),
            'new_item' => esc_html__('نظر جدید', 'mohtavanegar'),
            'view_item' => esc_html__('مشاهده نظر', 'mohtavanegar'),
            'search_items' => esc_html__('جستجوی نظرات', 'mohtavanegar'),
            'not_found' => esc_html__('نظری یافت نشد', 'mohtavanegar'),
            'not_found_in_trash' => esc_html__('نظری در سطل زباله یافت نشد', 'mohtavanegar'),
        ];

        $args = [
            'labels' => $labels,
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => false,
            'capability_type' => 'post',
            'hierarchical' => false,
            'supports' => ['title', 'editor', 'author'],
            'has_archive' => false,
            'menu_icon' => 'dashicons-star-filled',
        ];

        register_post_type('patient_review', $args);
    }

    public function add_reviews_admin_menu() {
        $pending_count = $this->get_pending_reviews_count();
        $menu_title = esc_html__('نظر بیماران', 'mohtavanegar');
        if ($pending_count > 0) {
            $pending_count = (int) $pending_count;
            $menu_title .= ' <span class="update-plugins count-' . $pending_count . '"><span class="plugin-count">' . $pending_count . '</span></span>';
        }
        add_menu_page(
            esc_html__('نظر بیماران', 'mohtavanegar'),
            $menu_title,
            'manage_options',
            'patient-reviews',
            [$this, 'render_reviews_admin_page'],
            'dashicons-star-filled',
            26
        );
    }

    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'toplevel_page_patient-reviews') {
            return;
        }
        wp_enqueue_script(
            'reviews-admin-js',
            get_template_directory_uri() . '/js/reviews-admin.js',
            ['jquery'],
            '1.0.0',
            true
        );
        wp_localize_script('reviews-admin-js', 'reviewsAdminData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('reviews_admin_nonce'),
        ]);
    }

    private function get_reviews($status = 'pending') {
        $args = [
            'post_type' => 'patient_review',
            'post_status' => $status,
            'posts_per_page' => -1,
            'orderby' => 'date',
            'order' => 'DESC',
        ];
        return get_posts($args);
    }

    public function render_reviews_admin_page() {
        $pending = $this->get_reviews('pending');
        $approved = $this->get_reviews('publish');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('مدیریت نظر بیماران', 'mohtavanegar'); ?></h1>

            <h2 class="nav-tab-wrapper">
                <a href="#pending-reviews" class="nav-tab nav-tab-active"><?php echo esc_html__('در انتظار تایید', 'mohtavanegar'); ?></a>
                <a href="#approved-reviews" class="nav-tab"><?php echo esc_html__('تایید شده', 'mohtavanegar'); ?></a>
            </h2>

            <div id="pending-reviews" class="tab-content">
                <h3><?php echo esc_html__('نظرات در انتظار تایید', 'mohtavanegar'); ?></h3>
                <?php if (empty($pending)) : ?>
                    <p><?php echo esc_html__('موردی یافت نشد.', 'mohtavanegar'); ?></p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('نام', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('خدمت', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('امتیاز', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('متن نظر', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('تاریخ', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('عملیات', 'mohtavanegar'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($pending as $item) : ?>
                            <tr id="review-<?php echo esc_attr($item->ID); ?>">
                                <td><?php echo esc_html(get_post_meta($item->ID, '_review_name', true)); ?></td>
                                <td>
                                    <?php
                                    $service_id = get_post_meta($item->ID, '_review_service', true);
                                    if ($service_id) {
                                        $service = get_post($service_id);
                                        echo esc_html($service ? $service->post_title : '-');
                                    } else {
                                        echo '-';
                                    }
                                    ?>
                                </td>
                                <td><?php echo intval(get_post_meta($item->ID, '_review_rating', true)); ?></td>
                                <td><?php echo esc_html($item->post_content); ?></td>
                                <td><?php echo get_the_date('Y-m-d H:i:s', $item->ID); ?></td>
                                <td>
                                    <button class="button approve-review" data-id="<?php echo esc_attr($item->ID); ?>"><?php echo esc_html__('تایید', 'mohtavanegar'); ?></button>
                                    <button class="button delete-review" data-id="<?php echo esc_attr($item->ID); ?>"><?php echo esc_html__('حذف', 'mohtavanegar'); ?></button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

            <div id="approved-reviews" class="tab-content" style="display:none;">
                <h3><?php echo esc_html__('نظرات تایید شده', 'mohtavanegar'); ?></h3>
                <?php if (empty($approved)) : ?>
                    <p><?php echo esc_html__('موردی یافت نشد.', 'mohtavanegar'); ?></p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php echo esc_html__('نام', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('خدمت', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('امتیاز', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('متن نظر', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('تاریخ', 'mohtavanegar'); ?></th>
                                <th><?php echo esc_html__('عملیات', 'mohtavanegar'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($approved as $item) : ?>
                            <tr id="review-<?php echo esc_attr($item->ID); ?>">
                                <td><?php echo esc_html(get_post_meta($item->ID, '_review_name', true)); ?></td>
                                <td>
                                    <?php
                                    $service_id = get_post_meta($item->ID, '_review_service', true);
                                    if ($service_id) {
                                        $service = get_post($service_id);
                                        echo esc_html($service ? $service->post_title : '-');
                                    } else {
                                        echo '-';
                                    }
                                    ?>
                                </td>
                                <td><?php echo intval(get_post_meta($item->ID, '_review_rating', true)); ?></td>
                                <td><?php echo esc_html($item->post_content); ?></td>
                                <td><?php echo get_the_date('Y-m-d H:i:s', $item->ID); ?></td>
                                <td>
                                    <button class="button delete-review" data-id="<?php echo esc_attr($item->ID); ?>"><?php echo esc_html__('حذف', 'mohtavanegar'); ?></button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public function submit_patient_review() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'reviews_submit_nonce')) {
            wp_send_json_error('Invalid nonce');
        }

        $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
        $service_raw = isset($_POST['service']) ? sanitize_text_field(wp_unslash($_POST['service'])) : '';
        $rating = isset($_POST['rating']) ? intval($_POST['rating']) : 0;
        $text = isset($_POST['text']) ? wp_kses_post($_POST['text']) : '';

        $service_id = 0;
        $service_type = '';
        if (!empty($service_raw)) {
            if (preg_match('/^(service|category)-(\d+)$/', $service_raw, $matches)) {
                $service_type = $matches[1];
                $service_id = intval($matches[2]);
            } elseif (ctype_digit($service_raw)) {
                // backward compatibility: plain numeric (assume service post ID)
                $service_type = 'service';
                $service_id = intval($service_raw);
            }
        }

        if (empty($name) || empty($text) || $rating < 1 || $rating > 5) {
            wp_send_json_error(esc_html__('لطفاً نام، متن نظر و امتیاز معتبر وارد کنید.', 'mohtavanegar'));
        }

        $post_id = wp_insert_post([
            'post_title' => $name,
            'post_content' => $text,
            'post_type' => 'patient_review',
            'post_status' => 'pending',
            'post_author' => 1,
        ]);

        if ($post_id && !is_wp_error($post_id)) {
            update_post_meta($post_id, '_review_name', $name);
            update_post_meta($post_id, '_review_service', $service_id);
            update_post_meta($post_id, '_review_service_type', $service_type);
            update_post_meta($post_id, '_review_rating', $rating);
            wp_send_json_success(esc_html__('نظر شما ثبت شد و پس از تایید نمایش داده می‌شود.', 'mohtavanegar'));
        }

        wp_send_json_error(esc_html__('خطا در ثبت نظر. دوباره تلاش کنید.', 'mohtavanegar'));
    }

    public function approve_patient_review() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'reviews_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        $id = isset($_POST['review_id']) ? intval($_POST['review_id']) : 0;
        if ($id <= 0) {
            wp_send_json_error('Invalid ID');
        }
        $updated = wp_update_post([
            'ID' => $id,
            'post_status' => 'publish',
        ]);
        if ($updated) {
            wp_send_json_success('Approved');
        }
        wp_send_json_error('Failed');
    }

    public function delete_patient_review() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'reviews_admin_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        $id = isset($_POST['review_id']) ? intval($_POST['review_id']) : 0;
        if ($id <= 0) {
            wp_send_json_error('Invalid ID');
        }
        $deleted = wp_delete_post($id, true);
        if ($deleted) {
            wp_send_json_success('Deleted');
        }
        wp_send_json_error('Failed');
    }
}

new Mohtavanegar_Reviews_System();
