<?php

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

if (!defined('ABSPATH')) {
    exit;
}

class Mohtavanegar_Taxonomy_Display_Widget extends Widget_Base {
    public function get_style_depends() {
        return ['taxonomy-display-widget'];
    }
    public function get_name() {
        return 'mohtavanegar-taxonomy-display';
    }

    public function get_title() {
        return __('نمایش دسته بندی', 'mohtavanegar');
    }

    public function get_icon() {
        return 'eicon-post-list';
    }

    public function get_categories() {
        return ['general'];
    }

    protected function register_controls() {
        // Content Settings
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('تنظیمات محتوا', 'mohtavanegar'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'taxonomy_type',
            [
                'label' => __('نوع دسته بندی', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'category' => 'دسته بندی مقالات',
                    'post_tag' => 'برچسب ها',
                    'services_category' => 'دسته بندی خدمات',
                    'portfolio_category' => 'دسته بندی نمونه کارها',
                    'sellino_product_category' => 'دسته بندی محصولات',
                ],
                'default' => 'category',
            ]
        );

        $this->add_control(
            'display_type',
            [
                'label' => __('نوع نمایش', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'list' => 'لیستی',
                    'grid' => 'شبکه ای',
                ],
                'default' => 'list',
            ]
        );

        // Desktop Columns
        $this->add_control(
            'grid_columns',
            [
                'label' => __('تعداد ستون ها (دسکتاپ)', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => '1 ستون',
                    '2' => '2 ستون',
                    '3' => '3 ستون',
                    '4' => '4 ستون',
                    '5' => '5 ستون',
                    '6' => '6 ستون',
                ],
                'default' => '3',
                'condition' => [
                    'display_type' => 'grid',
                ],
                'selectors' => [
                    '(desktop){{WRAPPER}} .taxonomy-display' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
            ]
        );

        // Tablet Columns
        $this->add_control(
            'grid_columns_tablet',
            [
                'label' => __('تعداد ستون ها (تبلت)', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => __('پیش فرض', 'mohtavanegar'),
                    '1' => '1 ستون',
                    '2' => '2 ستون',
                    '3' => '3 ستون',
                    '4' => '4 ستون',
                ],
                'default' => '2',
                'condition' => [
                    'display_type' => 'grid',
                ],
                'selectors' => [
                    '(tablet){{WRAPPER}} .taxonomy-display' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
            ]
        );

        // Mobile Columns
        $this->add_control(
            'grid_columns_mobile',
            [
                'label' => __('تعداد ستون ها (موبایل)', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => __('پیش فرض', 'mohtavanegar'),
                    '1' => '1 ستون',
                    '2' => '2 ستون',
                ],
                'default' => '1',
                'condition' => [
                    'display_type' => 'grid',
                ],
                'selectors' => [
                    '(mobile){{WRAPPER}} .taxonomy-display' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
            ]
        );

        // Text Alignment
        $this->add_control(
            'text_align',
            [
                'label' => __('چیدمان متن', 'mohtavanegar'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('چپ', 'mohtavanegar'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('وسط', 'mohtavanegar'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('راست', 'mohtavanegar'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-item' => 'text-align: {{VALUE}};',
                ],
                'default' => 'right',
            ]
        );

        // Title Color
        $this->add_control(
            'title_color',
            [
                'label' => __('رنگ عنوان', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'default' => '#0073aa',
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-item .taxonomy-link' => 'color: {{VALUE}};',
                ],
            ]
        );

        // Icon Size
        $this->add_control(
            'icon_size',
            [
                'label' => __('سایز آیکون', 'mohtavanegar'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 300,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 40,
                ],
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-icon img' => 'width: {{SIZE}}{{UNIT}}; height: auto; max-width: 100%;',
                ],
                'condition' => [
                    'show_icon' => 'yes',
                ],
            ]
        );

        // Icon Border Radius
        $this->add_control(
            'icon_border_radius',
            [
                'label' => __('گرد بودن آیکون', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'allowed_dimensions' => ['all', 'top', 'right', 'bottom', 'left'],
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-icon img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '5',
                    'right' => '5',
                    'bottom' => '5',
                    'left' => '5',
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'condition' => [
                    'show_icon' => 'yes',
                ],
            ]
        );

        // Item Border Radius
        $this->add_control(
            'item_border_radius',
            [
                'label' => __('گرد بودن پس زمینه آیتم', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'allowed_dimensions' => ['all', 'top', 'right', 'bottom', 'left'],
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '8',
                    'right' => '8',
                    'bottom' => '8',
                    'left' => '8',
                    'unit' => 'px',
                    'isLinked' => true,
                ],
            ]
        );

        // Grid Spacing
        $this->add_responsive_control(
            'grid_spacing',
            [
                'label' => __('فاصله بین سطر و ستون', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-display' => 'gap: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '20',
                    'right' => '20',
                    'unit' => 'px',
                ],
                'condition' => [
                    'display_type' => 'grid',
                ],
            ]
        );

        $this->add_control(
            'show_count',
            [
                'label' => __('نمایش تعداد زیر شاخه', 'mohtavanegar'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_icon',
            [
                'label' => __('نمایش آیکون', 'mohtavanegar'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Add icon size selection
        $this->add_control(
            'icon_size_select',
            [
                'label' => __('سایز آیکون', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'thumbnail' => __('تصویر کوچک', 'mohtavanegar'),
                    'medium' => __('متوسط', 'mohtavanegar'),
                    'large' => __('بزرگ', 'mohtavanegar'),
                    'full' => __('سایز کامل', 'mohtavanegar'),
                ],
                'default' => 'thumbnail',
                'condition' => [
                    'show_icon' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'show_only_parents',
            [
                'label' => __('فقط دسته بندی های اصلی', 'mohtavanegar'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'description' => __('فقط سرشاخه ها نمایش داده شوند', 'mohtavanegar'),
            ]
        );

        $this->add_control(
            'icon_position',
            [
                'label' => __('موقعیت آیکون', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'right' => 'راست نوشته',
                    'left' => 'چپ نوشته',
                    'top' => 'بالای نوشته و وسط چین',
                    'bottom' => 'پایین نوشته و وسط چین',
                ],
                'default' => 'right',
                'condition' => [
                    'show_icon' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'order_by',
            [
                'label' => __('ترتیب نمایش', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'asc' => __('صعودی (بر اساس ID)', 'mohtavanegar'),
                    'desc' => __('نزولی (بر اساس ID)', 'mohtavanegar'),
                    'alpha' => __('حروف الفبا', 'mohtavanegar'),
                    'alpha_reverse' => __('حروف الفبا برعکس', 'mohtavanegar'),
                    'manual' => __('سفارشی (دستی)', 'mohtavanegar'),
                ],
                'default' => 'asc',
            ]
        );

        // Manual order repeater
        $repeater = new \Elementor\Repeater();
        $repeater->add_control(
            'manual_term',
            [
                'label' => __('انتخاب دسته', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT2,
                'options' => $this->get_all_terms_for_select(),
                'multiple' => false,
            ]
        );
        $this->add_control(
            'manual_order',
            [
                'label' => __('ترتیب دستی دسته‌ها', 'mohtavanegar'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [],
                'condition' => [
                    'order_by' => 'manual',
                ],
            ]
        );

        $this->add_control(
            'exclude_terms',
            [
                'label' => __('دسته‌بندی‌های مستثنی', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT2,
                'options' => $this->get_all_terms_for_select(),
                'multiple' => true,
                'label_block' => true,
                'description' => __('دسته‌بندی‌هایی که نمی‌خواهید نمایش داده شوند را انتخاب کنید', 'mohtavanegar'),
            ]
        );

        $this->end_controls_section();

        // Style Settings
        $this->start_controls_section(
            'style_section',
            [
                'label' => __('تنظیمات ظاهری', 'mohtavanegar'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => __('رنگ متن', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-item' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => __('پس زمینه کادر', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-item' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        // Item Padding
        $this->add_responsive_control(
            'item_padding',
            [
                'label' => __('فاصله داخلی آیتم', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .taxonomy-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '12',
                    'right' => '16',
                    'bottom' => '12',
                    'left' => '16',
                    'unit' => 'px',
                    'isLinked' => false,
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'selector' => '{{WRAPPER}} .taxonomy-item',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'box_shadow',
                'selector' => '{{WRAPPER}} .taxonomy-item',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'selector' => '{{WRAPPER}} .taxonomy-item',
            ]
        );

        // List marker controls (for list display)
        $this->add_control(
            'show_list_marker',
            [
                'label' => __('نمایش بولت لیست', 'mohtavanegar'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'condition' => [
                    'display_type' => 'list',
                ],
            ]
        );

        $this->add_control(
            'list_marker_text',
            [
                'label' => __('کاراکتر بولت', 'mohtavanegar'),
                'type' => Controls_Manager::TEXT,
                'default' => '•',
                'placeholder' => '* , • , –',
                'condition' => [
                    'display_type' => 'list',
                    'show_list_marker' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'list_marker_color',
            [
                'label' => __('رنگ بولت', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .display-list .taxonomy-item .list-marker' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'display_type' => 'list',
                    'show_list_marker' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'list_marker_size',
            [
                'label' => __('سایز بولت', 'mohtavanegar'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [ 'min' => 8, 'max' => 48, 'step' => 1 ],
                    'em' => [ 'min' => 0.5, 'max' => 3, 'step' => 0.1 ],
                    'rem' => [ 'min' => 0.5, 'max' => 3, 'step' => 0.1 ],
                ],
                'default' => [ 'size' => 14, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .display-list .taxonomy-item .list-marker' => 'font-size: {{SIZE}}{{UNIT}}; line-height: 1;',
                ],
                'condition' => [
                    'display_type' => 'list',
                    'show_list_marker' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'list_marker_spacing',
            [
                'label' => __('فاصله بولت تا متن', 'mohtavanegar'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [ 'min' => 0, 'max' => 40, 'step' => 1 ],
                    'em' => [ 'min' => 0, 'max' => 3, 'step' => 0.1 ],
                ],
                'default' => [ 'size' => 8, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .display-list .taxonomy-item .list-marker' => 'margin-inline-end: {{SIZE}}{{UNIT}}; display: inline-block;',
                ],
                'condition' => [
                    'display_type' => 'list',
                    'show_list_marker' => 'yes',
                ],
            ]
        );

        // List gap (when list display is selected)
        $this->add_responsive_control(
            'list_gap',
            [
                'label' => __('فاصله آیتم‌های لیست', 'mohtavanegar'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range' => [
                    'px' => [ 'min' => 0, 'max' => 100, 'step' => 1 ],
                    'em' => [ 'min' => 0, 'max' => 5, 'step' => 0.1 ],
                    '%'  => [ 'min' => 0, 'max' => 10, 'step' => 0.5 ],
                ],
                'default' => [ 'size' => 12, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .display-list .taxonomy-item + .taxonomy-item' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'display_type' => 'list',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $args = [
            'taxonomy' => $settings['taxonomy_type'],
            'hide_empty' => false,
        ];
        
        // Filter to show only parent categories if the option is enabled
        if ($settings['show_only_parents'] === 'yes') {
            $args['parent'] = 0; // Only get terms with no parent (main categories)
        }

        // Manual order logic
        if ($settings['order_by'] === 'manual' && !empty($settings['manual_order'])) {
            $manual_ids = array_column($settings['manual_order'], 'manual_term');
            $args['include'] = $manual_ids;
            $args['orderby'] = 'include';
        } else {
            // Apply order by
            switch ($settings['order_by']) {
                case 'asc':
                    $args['orderby'] = 'id';
                    $args['order'] = 'ASC';
                    break;
                case 'desc':
                    $args['orderby'] = 'id';
                    $args['order'] = 'DESC';
                    break;
                case 'alpha':
                    $args['orderby'] = 'name';
                    $args['order'] = 'ASC';
                    break;
                case 'alpha_reverse':
                    $args['orderby'] = 'name';
                    $args['order'] = 'DESC';
                    break;
                default:
                    $args['orderby'] = 'id';
                    $args['order'] = 'ASC';
            }
        }
        
        // Exclude terms if specified
        if (!empty($settings['exclude_terms'])) {
            $args['exclude'] = $settings['exclude_terms'];
        }
        
        $terms = get_terms($args);

        if (empty($terms)) {
            return;
        }

        $this->add_render_attribute('wrapper', 'class', 'taxonomy-display');
        $this->add_render_attribute('wrapper', 'class', 'display-' . $settings['display_type']);
        
        if ($settings['display_type'] === 'grid') {
            // Add base grid styles
            $this->add_render_attribute('wrapper', 'class', 'elementor-grid');
            
            // Add responsive data attributes with defaults
            $grid_columns = isset($settings['grid_columns']) ? $settings['grid_columns'] : '3';
            $grid_columns_tablet = isset($settings['grid_columns_tablet']) ? $settings['grid_columns_tablet'] : '2';
            $grid_columns_mobile = isset($settings['grid_columns_mobile']) ? $settings['grid_columns_mobile'] : '1';
            
            $this->add_render_attribute('wrapper', 'data-grid-columns', $grid_columns);
            $this->add_render_attribute('wrapper', 'data-grid-columns-tablet', $grid_columns_tablet);
            $this->add_render_attribute('wrapper', 'data-grid-columns-mobile', $grid_columns_mobile);
            
            // Inline styles for grid
            $this->add_render_attribute('wrapper', 'style', 'display: grid; gap: 20px;');
        }

        // Text alignment class
        $text_align_class = '';
        if (!empty($settings['text_align'])) {
            $text_align_class = 'text-align-' . $settings['text_align'];
        }

        ?>
        <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
            <?php foreach ($terms as $term) : 
                $icon_id = get_term_meta($term->term_id, 'taxonomy_icon', true);
                $icon_position = $settings['icon_position'];
                $has_icon = ($settings['show_icon'] === 'yes' && $icon_id);
                
                // Item class based on icon position
                $item_class = 'taxonomy-item ' . $text_align_class;
                if ($has_icon) {
                    $item_class .= ' has-icon icon-' . $icon_position;
                }
            ?>
                <div class="<?php echo esc_attr($item_class); ?>">
                    <?php if ($has_icon && ($icon_position === 'top' || $icon_position === 'left')) : ?>
                        <div class="taxonomy-icon position-<?php echo esc_attr($icon_position); ?>">
                            <a href="<?php echo get_term_link($term); ?>" class="taxonomy-link">
                                <?php echo wp_get_attachment_image($icon_id, $settings['icon_size_select'] ?: 'thumbnail'); ?>
                            </a>
                        </div>
                    <?php endif; ?>

                    <div class="taxonomy-content">
                        <a href="<?php echo get_term_link($term); ?>" class="taxonomy-link">
                            <?php if ($settings['display_type'] === 'list' && isset($settings['show_list_marker']) && $settings['show_list_marker'] === 'yes') : ?>
                                <span class="list-marker" aria-hidden="true"><?php echo esc_html($settings['list_marker_text'] ?? '•'); ?></span>
                            <?php endif; ?>
                            <span class="taxonomy-name"><?php echo esc_html($term->name); ?></span>
                            <?php if ($settings['show_count'] === 'yes') : ?>
                                <span class="term-count">(<?php echo $term->count; ?>)</span>
                            <?php endif; ?>
                        </a>
                    </div>

                    <?php if ($has_icon && ($icon_position === 'bottom' || $icon_position === 'right')) : ?>
                        <div class="taxonomy-icon position-<?php echo esc_attr($icon_position); ?>">
                            <a href="<?php echo get_term_link($term); ?>" class="taxonomy-link">
                                <?php echo wp_get_attachment_image($icon_id, $settings['icon_size_select'] ?: 'thumbnail'); ?>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
    }

    // Helper function to get all terms for select2
    private function get_all_terms_for_select() {
        $taxonomies = [
            'category',
            'post_tag',
            'services_category',
            'portfolio_category',
        ];
        $options = [];
        foreach ($taxonomies as $tax) {
            $terms = get_terms([
                'taxonomy' => $tax,
                'hide_empty' => false,
            ]);
            foreach ($terms as $term) {
                $options[$term->term_id] = $term->name . ' (' . $tax . ')';
            }
        }
        return $options;
    }
}

// Register the widget
add_action('elementor/widgets/register', function($widgets_manager) {
    $widgets_manager->register(new Mohtavanegar_Taxonomy_Display_Widget());
});
