(function ($) {
    'use strict';

    // Aparat Playlist Widget JavaScript
    class AparatPlaylistWidget {
        constructor() {
            this.init();
        }

        init() {
            // Wait for DOM to be ready
            $(document).ready(() => {
                this.setupVideoWrappers();
                this.setupAccessibility();
                this.setupErrorHandling();
            });
        }

        setupVideoWrappers() {
            // Find all video wrappers and setup proper aspect ratios
            $('.aparat-video-wrapper').each(function () {
                const $wrapper = $(this);
                const $iframe = $wrapper.find('iframe');

                if ($iframe.length) {
                    // Ensure iframe has proper attributes
                    $iframe.attr({
                        'allowfullscreen': 'true',
                        'allow': 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture'
                    });
                }
            });
        }


        setupAccessibility() {
            // Add keyboard navigation for video items
            $('.aparat-item').attr('tabindex', '0');

            $('.aparat-item').on('keydown', function (e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    const $iframe = $(this).find('iframe').first();
                    if ($iframe.length) {
                        // Focus on the iframe
                        $iframe.focus();
                    }
                }
            });

            // Add ARIA labels for better accessibility
            $('.aparat-video-wrapper').each(function (index) {
                const $wrapper = $(this);
                const $description = $wrapper.siblings('.aparat-description');
                const description = $description.text().trim();

                if (description) {
                    $wrapper.attr('aria-label', `ویدیو ${index + 1}: ${description}`);
                } else {
                    $wrapper.attr('aria-label', `ویدیو ${index + 1}`);
                }
            });
        }

        setupErrorHandling() {
            // Monitor iframe loading
            // We use event delegation since iframes are added dynamically
            $(document).on('load', '.aparat-video-wrapper iframe', function () {
                const $wrapper = $(this).closest('.aparat-video-wrapper');
                $wrapper.removeClass('loading error');
            });

            // Add loading state initially to all wrappers
            $('.aparat-video-wrapper').each(function () {
                const $wrapper = $(this);
                // If it already has an iframe (e.g. re-init in editor), remove loading
                if ($wrapper.find('iframe').length) {
                    $wrapper.removeClass('loading error');
                } else {
                    $wrapper.addClass('loading');
                }

                // Set a timeout to check if it's still loading after 10 seconds
                // This handles cases where the script fails to load
                setTimeout(() => {
                    if ($wrapper.hasClass('loading') && !$wrapper.find('iframe').length) {
                        $wrapper.removeClass('loading').addClass('error').html(
                            '<div class="error-message">خطا در بارگذاری ویدیو. لطفاً اتصال اینترنت خود را بررسی کنید.</div>'
                        );
                    }
                }, 10000);
            });
        }

        // Utility function to validate Aparat URLs
        static validateAparatUrl(url) {
            if (!url) return false;

            const aparatRegex = /aparat\.com\/v\/([a-zA-Z0-9_-]+)/i;
            return aparatRegex.test(url) || /^[a-zA-Z0-9_-]{7,15}$/.test(url);
        }

        // Utility function to extract video ID
        static extractVideoId(url) {
            if (!url) return '';

            const match = url.match(/aparat\.com\/v\/([a-zA-Z0-9_-]+)/i);
            if (match) return match[1];

            if (/^[a-zA-Z0-9_-]{7,15}$/.test(url)) return url;

            return '';
        }
    }

    // Initialize the widget
    const aparatPlaylistWidget = new AparatPlaylistWidget();

    // Make it available globally for potential external use
    window.AparatPlaylistWidget = AparatPlaylistWidget;

    // Remove loading spinner after page fully loads
    $(window).on('load', function () {
        $('.aparat-video-wrapper.loading').each(function () {
            this.style.setProperty('content', 'none', 'important');
        });
    });

    // Handle Elementor editor preview
    if (typeof window.elementorFrontend !== 'undefined' &&
        typeof window.elementorFrontend.hooks !== 'undefined' &&
        typeof window.elementorFrontend.hooks.addAction === 'function') {
        elementorFrontend.hooks.addAction('frontend/element_ready/aparat_playlist_widget.default', function ($scope) {
            // Re-initialize when widget is loaded in Elementor editor
            new AparatPlaylistWidget();
        });
    }

})(jQuery);

// Additional utility functions for potential external use
window.AparatUtils = {
    // Generate embed code dynamically
    generateEmbedCode: function (videoUrl) {
        const videoId = AparatPlaylistWidget.extractVideoId(videoUrl);
        if (!videoId) return '';

        const randomDivId = 'aparat_' + Math.random().toString(36).substr(2, 9);
        return `<div id="${randomDivId}"><script type="text/JavaScript" src="https://www.aparat.com/embed/${videoId}?data[rnddiv]=${randomDivId}&data[responsive]=yes"></script></div>`;
    },

    // Check if video is valid
    isValidVideo: function (videoUrl) {
        return AparatPlaylistWidget.validateAparatUrl(videoUrl);
    },

    // Get video thumbnail URL (if needed in future)
    getThumbnailUrl: function (videoUrl) {
        const videoId = AparatPlaylistWidget.extractVideoId(videoUrl);
        if (!videoId) return '';

        // Aparat typically uses this pattern for thumbnails
        return `https://www.aparat.com/videothumb/embed/${videoId}.jpg`;
    }
};
