(function($){
  'use strict';

  const MN = window.MN_SEARCH_MODAL_VARS || {};

  function debounce(fn, wait){
    let t; return function(){
      const ctx=this, args=arguments; clearTimeout(t);
      t=setTimeout(function(){ fn.apply(ctx,args); }, wait);
    };
  }

  function buildSearchUrl(q, postTypes){
    const base = MN.home_url || '/';
    const url = new URL(base, window.location.origin);
    url.searchParams.set('s', q || '');
    if (Array.isArray(postTypes)){
      postTypes.forEach(pt => url.searchParams.append('post_type[]', pt));
    }
    return url.toString();
  }

  function renderResults($wrap, resp){
    const $list = $wrap.find('.mn-search-modal__results');
    const $loading = $wrap.find('.mn-search-modal__loading');
    const $more = $wrap.find('.mn-search-modal__more');

    $loading.attr('hidden', true);
    $list.empty();
    $more.attr('hidden', true).attr('href', '#');

    if(!resp || !resp.success){
      $list.append('<li class="mn-search-modal__empty">'+ (MN.i18n_error || 'Error') +'</li>');
      return;
    }

    const items = resp.data.items || [];
    const total = resp.data.total || 0;

    if(items.length === 0){
      $list.append('<li class="mn-search-modal__empty">'+(MN.i18n_empty || 'No results')+'</li>');
      return;
    }

    const first10 = items.slice(0, 10);
    first10.forEach(it => {
      const li = $('<li class="mn-search-modal__item" />');
      const a = $('<a target="_self"/>').attr('href', it.link).text(it.title || '');
      const ex = it.excerpt ? $('<div class="mn-search-modal__excerpt"/>').text(it.excerpt) : null;
      li.append(a);
      if (ex) li.append(ex);
      $list.append(li);
    });

    if(total > 10){
      const q = $wrap.data('q') || '';
      const postTypes = ($wrap.data('settings')||{}).postTypes || [];
      $more.attr('href', buildSearchUrl(q, postTypes)).attr('hidden', false);
    }
  }

  function doSearch($wrap, q){
    const settings = $wrap.data('settings') || {};
    const $loading = $wrap.find('.mn-search-modal__loading');
    const $list = $wrap.find('.mn-search-modal__results');

    $wrap.data('q', q);
    $loading.removeAttr('hidden');
    $list.empty();

    $.ajax({
      url: MN.ajax_url,
      method: 'POST',
      dataType: 'json',
      data: {
        action: 'mn_search_modal',
        nonce: MN.nonce,
        q: q,
        post_types: settings.postTypes || ['post']
      }
    }).done(function(resp){
      renderResults($wrap, resp);
    }).fail(function(){
      renderResults($wrap, { success:false });
    });
  }

  function openModal($wrap){
    const $overlay = $wrap.find('.mn-search-modal__overlay');
    const $toggle = $wrap.find('.mn-search-modal__toggle');
    $overlay.attr('aria-hidden','false').addClass('is-open');
    $toggle.attr('aria-expanded','true');
    setTimeout(()=>{ $wrap.find('.mn-search-modal__input').trigger('focus'); }, 50);
  }

  function closeModal($wrap){
    const $overlay = $wrap.find('.mn-search-modal__overlay');
    const $toggle = $wrap.find('.mn-search-modal__toggle');
    $overlay.attr('aria-hidden','true').removeClass('is-open');
    $toggle.attr('aria-expanded','false').trigger('focus');
  }

  function bind($scope){
    const $wrap = $scope.find('.mn-search-modal');
    if(!$wrap.length) return;

    // Toggle open
    $wrap.on('click', '.mn-search-modal__toggle', function(e){
      e.preventDefault();
      openModal($wrap);
    });

    // Close
    $wrap.on('click', '.mn-search-modal__close', function(e){
      e.preventDefault();
      closeModal($wrap);
    });

    // Click outside closes
    $wrap.on('click', '.mn-search-modal__overlay', function(e){
      if($(e.target).is('.mn-search-modal__overlay')){
        closeModal($wrap);
      }
    });

    // ESC closes
    $(document).on('keydown.mnSearchModal', function(e){
      if(e.key === 'Escape'){
        closeModal($wrap);
      }
    });

    const debounced = debounce(function(e){
      const val = $(e.target).val();
      if(typeof val === 'string' && val.trim().length){
        doSearch($wrap, val.trim());
      } else {
        $wrap.find('.mn-search-modal__results').empty();
        $wrap.find('.mn-search-modal__more').attr('hidden', true);
      }
    }, 400);

    $wrap.on('input', '.mn-search-modal__input', debounced);
  }

  // Elementor hook
  $(window).on('elementor/frontend/init', function(){
    if (elementorFrontend && elementorFrontend.hooks){
      elementorFrontend.hooks.addAction('frontend/element_ready/search_modal.default', function($scope){
        bind($scope);
      });
    }
  });

})(jQuery);
