<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Elementor Data Processing Functions
 * 
 * These functions help process Elementor data during import/export
 * to ensure widgets and templates are properly handled.
 */

/**
 * Process Elementor data in content to update template IDs and widget references
 *
 * @param array $data Elementor data
 * @param array $template_id_mapping Mapping of old template IDs to new ones
 * @return array Updated Elementor data
 */
function mohtavanegar_process_elementor_content_data($data, $template_id_mapping = array()) {
    if (!is_array($data)) {
        return $data;
    }
    
    // If no template mapping is provided, try to get it from options
    if (empty($template_id_mapping)) {
        $template_id_mapping = get_option('mohtavanegar_template_id_mapping', array());
    }
    
    // Get media mapping if available
    $media_id_mapping = get_option('mohtavanegar_media_id_mapping', array());
    
    // Get available Elementor widgets to ensure we don't lose widget types
    static $available_widgets = null;
    if ($available_widgets === null && class_exists('\Elementor\Plugin')) {
        try {
            $available_widgets = array();
            $widgets_manager = \Elementor\Plugin::instance()->widgets_manager;
            $widget_types = $widgets_manager->get_widget_types();
            
            foreach ($widget_types as $widget_key => $widget) {
                $available_widgets[] = $widget_key;
            }
            
            // Add our custom widgets to ensure they're recognized
            $custom_widgets = array(
                'mohtavanegar-before-after',
                'mohtavanegar_working_hours',
                'mohtavanegar_qa',
                'mohtavanegar_services_list',
                'mohtavanegar_taxonomy_display'
            );
            
            $available_widgets = array_merge($available_widgets, $custom_widgets);
            error_log('Available Elementor widgets for processing: ' . implode(', ', $available_widgets));
        } catch (\Exception $e) {
            error_log('Error getting Elementor widgets: ' . $e->getMessage());
            // Fallback to common widget types
            $available_widgets = array(
                'common', 'heading', 'image', 'text-editor', 'video', 'button', 'divider', 'spacer',
                'image-box', 'google-maps', 'icon', 'icon-box', 'image-gallery', 'image-carousel',
                'icon-list', 'counter', 'progress', 'testimonial', 'tabs', 'accordion', 'toggle',
                'social-icons', 'alert', 'audio', 'shortcode', 'html', 'menu-anchor', 'sidebar',
                'navigation-menu', 'search-form', 'slides', 'form', 'posts', 'portfolio', 'template'
            );
        }
    }
    
    foreach ($data as $key => $value) {
        if (is_array($value)) {
            // Process all widget types to ensure they're properly imported
            if (isset($value['elType']) && $value['elType'] === 'widget' && isset($value['widgetType'])) {
                // Preserve the widget type to prevent conversion to text editor
                $widget_type = $value['widgetType'];
                
                // Add widget class name if missing (important for Elementor to recognize the widget)
                if (!isset($value['settings']['_css_classes'])) {
                    $data[$key]['settings']['_css_classes'] = 'elementor-widget-' . $widget_type;
                }
                
                // Make sure the widget has a valid ID
                if (!isset($value['settings']['_element_id'])) {
                    $data[$key]['settings']['_element_id'] = 'elementor-element-' . uniqid();
                }
                
                // Process widget-specific settings
                if (isset($value['settings'])) {
                    // Handle image widgets
                    if (in_array($widget_type, array('image', 'image-box', 'image-gallery', 'image-carousel'))) {
                        foreach ($value['settings'] as $setting_key => $setting_value) {
                            if (is_array($setting_value) && isset($setting_value['url'])) {
                                // Update image URL if we have a media mapping
                                if (isset($media_id_mapping[$setting_value['url']])) {
                                    $new_url = wp_get_attachment_url($media_id_mapping[$setting_value['url']]);
                                    if ($new_url) {
                                        $data[$key]['settings'][$setting_key]['url'] = $new_url;
                                    }
                                }
                            }
                        }
                    }
                    
                    // Handle template widget
                    if ($widget_type === 'template') {
                        if (isset($value['settings']['template_id']) && isset($template_id_mapping[$value['settings']['template_id']])) {
                            $data[$key]['settings']['template_id'] = $template_id_mapping[$value['settings']['template_id']];
                        }
                    }
                    
                    // Handle gallery widget
                    if ($widget_type === 'image-gallery') {
                        if (isset($value['settings']['gallery']) && is_array($value['settings']['gallery'])) {
                            foreach ($value['settings']['gallery'] as $gallery_key => $gallery_item) {
                                if (isset($gallery_item['url']) && isset($media_id_mapping[$gallery_item['url']])) {
                                    $new_url = wp_get_attachment_url($media_id_mapping[$gallery_item['url']]);
                                    if ($new_url) {
                                        $data[$key]['settings']['gallery'][$gallery_key]['url'] = $new_url;
                                    }
                                }
                            }
                        }
                    }
                    
                    // Handle carousel widget
                    if ($widget_type === 'image-carousel') {
                        if (isset($value['settings']['carousel']) && is_array($value['settings']['carousel'])) {
                            foreach ($value['settings']['carousel'] as $carousel_key => $carousel_item) {
                                if (isset($carousel_item['url']) && isset($media_id_mapping[$carousel_item['url']])) {
                                    $new_url = wp_get_attachment_url($media_id_mapping[$carousel_item['url']]);
                                    if ($new_url) {
                                        $data[$key]['settings']['carousel'][$carousel_key]['url'] = $new_url;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            
            // Process nested elements
            if (isset($value['elements']) && is_array($value['elements'])) {
                $data[$key]['elements'] = mohtavanegar_process_elementor_content_data($value['elements'], $template_id_mapping);
            }
            
            // Process other array values
            foreach ($value as $sub_key => $sub_value) {
                if (is_array($sub_value) && $sub_key !== 'elements') {
                    $data[$key][$sub_key] = mohtavanegar_process_elementor_content_data($sub_value, $template_id_mapping);
                }
            }
        }
    }
    
    return $data;
}
