<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Elementor Templates Import/Export Functions
 */

/**
 * Export Elementor templates
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_elementor_templates($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Check if Elementor is active
    if (!class_exists('\Elementor\Plugin')) {
        $result['status'] = 'error';
        $result['message'] = __('افزونه المنتور فعال نیست.', 'mohtavanegar');
        return $result;
    }
    
    // Create templates directory
    $templates_dir = $export_dir . '/elementor-templates';
    if (!is_dir($templates_dir)) {
        mkdir($templates_dir, 0755, true);
    }
    
    // Get all Elementor templates
    $templates = get_posts(array(
        'post_type' => 'elementor_library',
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'orderby' => 'title',
        'order' => 'ASC'
    ));
    
    $templates_data = array();
    foreach ($templates as $template) {
        // Get template meta
        $template_meta = get_post_meta($template->ID);
        $filtered_meta = array();
        
        // Filter out unnecessary meta
        foreach ($template_meta as $meta_key => $meta_value) {
            // Keep all Elementor-related meta keys
            if (substr($meta_key, 0, 1) === '_' && 
                strpos($meta_key, '_elementor_') === 0) {
                $filtered_meta[$meta_key] = maybe_unserialize($meta_value[0]);
            }
        }
        
        // Get template type and conditions
        $template_type = get_post_meta($template->ID, '_elementor_template_type', true);
        $template_conditions = get_post_meta($template->ID, '_elementor_conditions', true);
        
        // Get template thumbnail
        $thumbnail_id = get_post_thumbnail_id($template->ID);
        $thumbnail_url = '';
        if ($thumbnail_id) {
            $thumbnail_url = wp_get_attachment_url($thumbnail_id);
        }
        
        $templates_data[] = array(
            'ID' => $template->ID,
            'post_title' => $template->post_title,
            'post_content' => $template->post_content,
            'post_excerpt' => $template->post_excerpt,
            'post_date' => $template->post_date,
            'post_name' => $template->post_name,
            'post_status' => $template->post_status,
            'post_author' => $template->post_author,
            'meta' => $filtered_meta,
            'template_type' => $template_type,
            'template_conditions' => $template_conditions,
            'thumbnail_url' => $thumbnail_url
        );
        
        // Log template export
        error_log('Exporting Elementor template: ' . $template->post_title . ' (ID: ' . $template->ID . ')');
    }
    
    $templates_file = $templates_dir . '/templates.json';
    file_put_contents($templates_file, json_encode($templates_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($templates);
    $result['message'] = sprintf(__('%d قالب المنتور با موفقیت برون‌ریزی شدند.', 'mohtavanegar'), count($templates));
    
    return $result;
}

/**
 * Import Elementor templates
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_elementor_templates($import_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Check if Elementor is active
    if (!class_exists('\Elementor\Plugin')) {
        $result['status'] = 'error';
        $result['message'] = __('افزونه المنتور فعال نیست.', 'mohtavanegar');
        return $result;
    }
    
    $templates_dir = $import_dir . '/elementor-templates';
    
    // Check if templates directory exists
    if (!is_dir($templates_dir)) {
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری قالب‌های المنتور یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // Import templates
    $templates_file = $templates_dir . '/templates.json';
    if (file_exists($templates_file)) {
        $templates_data = json_decode(file_get_contents($templates_file), true);
        
        if ($templates_data) {
            $imported_count = 0;
            $template_id_mapping = array();
            
            // First pass: Create all templates
            foreach ($templates_data as $template_data) {
                // Skip the Default Kit template
                if ($template_data['template_type'] === 'kit' || $template_data['post_title'] === 'Default Kit') {
                    error_log('Skipping Default Kit template: ' . $template_data['post_title']);
                    continue;
                }
                
                // Check if template exists by slug
                $existing_template = get_page_by_path($template_data['post_name'], OBJECT, 'elementor_library');
                
                if ($existing_template) {
                    // Update existing template
                    $template_id = $existing_template->ID;
                    
                    wp_update_post(array(
                        'ID' => $template_id,
                        'post_title' => $template_data['post_title'],
                        'post_content' => $template_data['post_content'],
                        'post_excerpt' => $template_data['post_excerpt'],
                        'post_status' => $template_data['post_status']
                    ));
                } else {
                    // Create new template
                    $template_id = wp_insert_post(array(
                        'post_title' => $template_data['post_title'],
                        'post_content' => $template_data['post_content'],
                        'post_excerpt' => $template_data['post_excerpt'],
                        'post_date' => $template_data['post_date'],
                        'post_name' => $template_data['post_name'],
                        'post_status' => $template_data['post_status'],
                        'post_author' => get_current_user_id(),
                        'post_type' => 'elementor_library'
                    ));
                }
                
                if ($template_id && !is_wp_error($template_id)) {
                    $template_id_mapping[$template_data['ID']] = $template_id;
                }
            }
            
            // Second pass: Update meta and process Elementor data
            foreach ($templates_data as $template_data) {
                $template_id = $template_id_mapping[$template_data['ID']] ?? null;
                
                if ($template_id && !is_wp_error($template_id)) {
                    // Set template type
                    if (!empty($template_data['template_type'])) {
                        update_post_meta($template_id, '_elementor_template_type', $template_data['template_type']);
                    }
                    
                    // Set template conditions
                    if (!empty($template_data['template_conditions'])) {
                        update_post_meta($template_id, '_elementor_conditions', $template_data['template_conditions']);
                    }
                    
                    // Set meta
                    if (!empty($template_data['meta'])) {
                        foreach ($template_data['meta'] as $meta_key => $meta_value) {
                            // Update Elementor data - replace old IDs with new ones
                            if ($meta_key === '_elementor_data' && !empty($meta_value)) {
                                $elementor_data = json_decode($meta_value, true);
                                if ($elementor_data) {
                                    // Process Elementor data to update IDs
                                    $elementor_data = mohtavanegar_process_elementor_data($elementor_data, $template_id_mapping);
                                    update_post_meta($template_id, $meta_key, wp_slash(json_encode($elementor_data)));
                                } else {
                                    update_post_meta($template_id, $meta_key, $meta_value);
                                }
                            } else {
                                update_post_meta($template_id, $meta_key, $meta_value);
                            }
                        }
                    }
                    
                    // Set thumbnail if available
                    if (!empty($template_data['thumbnail_url'])) {
                        $attachment_id = mohtavanegar_get_attachment_id_by_url($template_data['thumbnail_url']);
                        if ($attachment_id) {
                            set_post_thumbnail($template_id, $attachment_id);
                        }
                    }
                    
                    $imported_count++;
                }
            }
            
            // Store the template ID mapping for use in other imports
            update_option('mohtavanegar_template_id_mapping', $template_id_mapping);
            
            // Clear Elementor cache
            if (class_exists('\Elementor\Plugin')) {
                \Elementor\Plugin::$instance->files_manager->clear_cache();
            }
            
            $result['count'] = $imported_count;
            $result['message'] = sprintf(__('%d قالب المنتور با موفقیت درون‌ریزی شدند.', 'mohtavanegar'), $imported_count);
        }
    }
    
    return $result;
}

/**
 * Process Elementor data to update template IDs
 *
 * @param array $data Elementor data
 * @param array $template_id_mapping Mapping of old template IDs to new ones
 * @return array Updated Elementor data
 */
function mohtavanegar_process_elementor_data($data, $template_id_mapping) {
    if (!is_array($data)) {
        return $data;
    }
    
    // If mohtavanegar_process_elementor_content_data function exists, use it instead
    // as it has more comprehensive widget handling
    if (function_exists('mohtavanegar_process_elementor_content_data')) {
        error_log('Using enhanced Elementor data processor for template import');
        return mohtavanegar_process_elementor_content_data($data, $template_id_mapping);
    }
    
    // Get media mapping if available
    $media_id_mapping = get_option('mohtavanegar_media_id_mapping', array());
    
    // Get available Elementor widgets to ensure we don't lose widget types
    static $available_widgets = null;
    if ($available_widgets === null && class_exists('\Elementor\Plugin')) {
        try {
            $available_widgets = array();
            $widgets_manager = \Elementor\Plugin::instance()->widgets_manager;
            $widget_types = $widgets_manager->get_widget_types();
            
            foreach ($widget_types as $widget_key => $widget) {
                $available_widgets[] = $widget_key;
            }
            
            error_log('Available Elementor widgets for template processing: ' . implode(', ', $available_widgets));
        } catch (\Exception $e) {
            error_log('Error getting Elementor widgets for template processing: ' . $e->getMessage());
            // Fallback to common widget types if we can't get them dynamically
            $available_widgets = array(
                'common', 'heading', 'image', 'text-editor', 'video', 'button', 'divider', 'spacer',
                'image-box', 'google-maps', 'icon', 'icon-box', 'image-gallery', 'image-carousel',
                'icon-list', 'counter', 'progress', 'testimonial', 'tabs', 'accordion', 'toggle',
                'social-icons', 'alert', 'audio', 'shortcode', 'html', 'menu-anchor', 'sidebar',
                'navigation-menu', 'search-form', 'slides', 'form', 'posts', 'portfolio', 'template',
                // Add our custom widgets to ensure they're recognized during import
                'mohtavanegar-before-after', 'mohtavanegar_working_hours', 'mohtavanegar_qa',
                'mohtavanegar_services_list', 'mohtavanegar_taxonomy_display'
            );
        }
    }
    
    foreach ($data as $key => $value) {
        if (is_array($value)) {
            // Process all widget types to ensure they're properly imported
            if (isset($value['elType']) && $value['elType'] === 'widget' && isset($value['widgetType'])) {
                // Preserve the widget type to prevent conversion to text editor
                $widget_type = $value['widgetType'];
                
                // Add widget class name if missing (important for Elementor to recognize the widget)
                if (!isset($value['settings']['_css_classes'])) {
                    $data[$key]['settings']['_css_classes'] = 'elementor-widget-' . $widget_type;
                }
                
                // Make sure the widget has a valid ID
                if (!isset($value['settings']['_element_id'])) {
                    $data[$key]['settings']['_element_id'] = 'elementor-element-' . uniqid();
                }
                
                // Check if this is a template widget
                if ($widget_type === 'template') {
                    // Update template ID if it exists in our mapping
                    if (isset($value['settings']['template_id']) && isset($template_id_mapping[$value['settings']['template_id']])) {
                        $data[$key]['settings']['template_id'] = $template_id_mapping[$value['settings']['template_id']];
                    }
                }
                
                // Process widget-specific settings (for common widget types)
                switch ($widget_type) {
                    case 'image':
                        // Handle image widget
                        if (isset($value['settings']['image']) && isset($value['settings']['image']['id']) && !empty($media_id_mapping)) {
                            // If we have a mapping for this image ID, update it
                            if (isset($media_id_mapping[$value['settings']['image']['id']])) {
                                $data[$key]['settings']['image']['id'] = $media_id_mapping[$value['settings']['image']['id']];
                            }
                        }
                        break;
                        
                    case 'image-gallery':
                        // Handle image gallery widget
                        if (isset($value['settings']['gallery']) && is_array($value['settings']['gallery']) && !empty($media_id_mapping)) {
                            foreach ($value['settings']['gallery'] as $gallery_key => $gallery_item) {
                                if (isset($gallery_item['id']) && isset($media_id_mapping[$gallery_item['id']])) {
                                    $data[$key]['settings']['gallery'][$gallery_key]['id'] = $media_id_mapping[$gallery_item['id']];
                                }
                            }
                        }
                        break;
                        
                    // Add more widget types as needed
                }
            }
            
            // Process nested elements
            if (isset($value['elements']) && is_array($value['elements'])) {
                $data[$key]['elements'] = mohtavanegar_process_elementor_data($value['elements'], $template_id_mapping);
            }
            
            // Process other nested arrays
            foreach ($value as $sub_key => $sub_value) {
                if (is_array($sub_value) && $sub_key !== 'elements') {
                    $data[$key][$sub_key] = mohtavanegar_process_elementor_data($sub_value, $template_id_mapping);
                }
            }
        }
    }
    
    return $data;
}

// Helper function to get attachment ID by URL
function mohtavanegar_get_attachment_id_by_url($url) {
    global $wpdb;
    
    $attachment_id = $wpdb->get_var($wpdb->prepare(
        "SELECT ID FROM $wpdb->posts WHERE guid = %s AND post_type = 'attachment'",
        $url
    ));
    
    return $attachment_id;
}
