<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Export Handler Functions
 */

/**
 * Start export process
 */
function mohtavanegar_start_export() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mohtavanegar_import_export_nonce')) {
        wp_send_json_error(array('message' => __('خطای امنیتی. لطفاً صفحه را رفرش کنید.', 'mohtavanegar')));
    }
    
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('شما دسترسی لازم برای انجام این کار را ندارید.', 'mohtavanegar')));
    }
    
    // Check export items
    if (!isset($_POST['export_items']) || empty($_POST['export_items'])) {
        wp_send_json_error(array('message' => __('لطفاً حداقل یک مورد را برای برون‌ریزی انتخاب کنید.', 'mohtavanegar')));
    }
    
    $export_items = $_POST['export_items'];
    
    // Create temporary directory
    $upload_dir = wp_upload_dir();
    $export_dir = $upload_dir['basedir'] . '/mohtavanegar-export-' . time();
    
    if (!is_dir($export_dir)) {
        mkdir($export_dir, 0755, true);
    }
    
    // Save export configuration
    $export_config = array(
        'id' => uniqid('export_'),
        'date' => current_time('mysql'),
        'items' => $export_items,
        'dir' => $export_dir,
        'progress' => 0,
        'current_item' => '',
        'is_complete' => false,
        'download_url' => ''
    );
    
    update_option('mohtavanegar_current_export', $export_config);
    
    wp_send_json_success(array(
        'export_id' => $export_config['id'],
        'message' => __('برون‌ریزی آغاز شد.', 'mohtavanegar')
    ));
}
add_action('wp_ajax_mohtavanegar_start_export', 'mohtavanegar_start_export');

/**
 * Process export progress
 */
function mohtavanegar_export_progress() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mohtavanegar_import_export_nonce')) {
        wp_send_json_error(array('message' => __('خطای امنیتی. لطفاً صفحه را رفرش کنید.', 'mohtavanegar')));
    }
    
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('شما دسترسی لازم برای انجام این کار را ندارید.', 'mohtavanegar')));
    }
    
    // Check export ID
    if (!isset($_POST['export_id'])) {
        wp_send_json_error(array('message' => __('شناسه برون‌ریزی نامعتبر است.', 'mohtavanegar')));
    }
    
    $export_id = $_POST['export_id'];
    $step = isset($_POST['step']) ? intval($_POST['step']) : 0;
    
    // Get export configuration
    $export_config = get_option('mohtavanegar_current_export');
    
    if (!$export_config || $export_config['id'] !== $export_id) {
        wp_send_json_error(array('message' => __('برون‌ریزی یافت نشد.', 'mohtavanegar')));
    }
    
    // If export is already complete, return the result
    if ($export_config['is_complete']) {
        wp_send_json_success(array(
            'progress' => 100,
            'current_item' => __('تکمیل شد', 'mohtavanegar'),
            'is_complete' => true,
            'download_url' => $export_config['download_url']
        ));
    }
    
    // Process export step
    $export_items = $export_config['items'];
    $total_items = count($export_items);
    $current_item_index = $step;
    
    // Check if all items are processed
    if ($current_item_index >= $total_items) {
        // Create ZIP file
        $zip_file = mohtavanegar_create_export_zip($export_config['dir']);
        
        if ($zip_file) {
            // Update export configuration
            $export_config['is_complete'] = true;
            $export_config['progress'] = 100;
            $export_config['current_item'] = __('تکمیل شد', 'mohtavanegar');
            $export_config['download_url'] = $zip_file;
            
            update_option('mohtavanegar_current_export', $export_config);
            
            wp_send_json_success(array(
                'progress' => 100,
                'current_item' => __('تکمیل شد', 'mohtavanegar'),
                'is_complete' => true,
                'download_url' => $zip_file
            ));
        } else {
            wp_send_json_error(array('message' => __('خطا در ایجاد فایل زیپ.', 'mohtavanegar')));
        }
    }
    
    // Process current item
    $current_item = $export_items[$current_item_index];
    $export_config['current_item'] = $current_item;
    
    // Calculate progress
    $progress = round(($current_item_index / $total_items) * 100);
    $export_config['progress'] = $progress;
    
    update_option('mohtavanegar_current_export', $export_config);
    
    // Export current item
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    switch ($current_item) {
        case 'posts':
            $result = mohtavanegar_export_posts($export_config['dir']);
            break;
        
        case 'pages':
            $result = mohtavanegar_export_pages($export_config['dir']);
            break;
        
        case 'services':
            $result = mohtavanegar_export_services($export_config['dir']);
            break;
        
        case 'menus':
            $result = mohtavanegar_export_menus($export_config['dir']);
            break;
        
        case 'theme_settings':
            $result = mohtavanegar_export_theme_settings($export_config['dir']);
            break;
        
        case 'elementor_templates':
            $result = mohtavanegar_export_elementor_templates($export_config['dir']);
            break;
        
        case 'media':
            $result = mohtavanegar_export_media($export_config['dir']);
            break;
    }
    
    // Update current item name for display
    $item_names = array(
        'posts' => __('نوشته‌ها و دسته‌بندی‌ها', 'mohtavanegar'),
        'pages' => __('برگه‌ها', 'mohtavanegar'),
        'services' => __('خدمات و دسته‌بندی‌ها', 'mohtavanegar'),
        'menus' => __('منوها', 'mohtavanegar'),
        'theme_settings' => __('تنظیمات قالب', 'mohtavanegar'),
        'elementor_templates' => __('قالب‌های المنتور', 'mohtavanegar'),
        'media' => __('رسانه‌ها', 'mohtavanegar')
    );
    
    $current_item_name = isset($item_names[$current_item]) ? $item_names[$current_item] : $current_item;
    
    wp_send_json_success(array(
        'progress' => $progress,
        'current_item' => $current_item_name,
        'is_complete' => false,
        'result' => $result
    ));
}
add_action('wp_ajax_mohtavanegar_export_progress', 'mohtavanegar_export_progress');

/**
 * Create ZIP file from export directory
 *
 * @param string $export_dir Directory to zip
 * @return string|false URL to download the ZIP file or false on failure
 */
function mohtavanegar_create_export_zip($export_dir) {
    // Create manifest file
    $manifest = array(
        'date' => current_time('mysql'),
        'site_url' => site_url(),
        'site_title' => get_bloginfo('name'),
        'theme' => wp_get_theme()->get('Name'),
        'version' => '1.0'
    );
    
    $manifest_file = $export_dir . '/manifest.json';
    file_put_contents($manifest_file, json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Create ZIP file
    $upload_dir = wp_upload_dir();
    $zip_file = $upload_dir['basedir'] . '/mohtavanegar-export-' . date('Y-m-d') . '.zip';
    
    // Remove existing ZIP file
    if (file_exists($zip_file)) {
        unlink($zip_file);
    }
    
    // Create new ZIP file
    $zip = new ZipArchive();
    if ($zip->open($zip_file, ZipArchive::CREATE) !== true) {
        return false;
    }
    
    // Add files to ZIP
    mohtavanegar_add_files_to_zip($zip, $export_dir, basename($export_dir));
    
    $zip->close();
    
    // Return download URL
    return $upload_dir['baseurl'] . '/mohtavanegar-export-' . date('Y-m-d') . '.zip';
}

/**
 * Add files to ZIP recursively
 *
 * @param ZipArchive $zip ZIP archive
 * @param string $dir Directory to add
 * @param string $zip_dir Directory name in ZIP
 */
function mohtavanegar_add_files_to_zip($zip, $dir, $zip_dir = '') {
    if (is_dir($dir)) {
        if ($handle = opendir($dir)) {
            while (($file = readdir($handle)) !== false) {
                if ($file != '.' && $file != '..') {
                    $file_path = $dir . '/' . $file;
                    $zip_path = $zip_dir . '/' . $file;
                    
                    if (is_file($file_path)) {
                        $zip->addFile($file_path, $zip_path);
                    } elseif (is_dir($file_path)) {
                        $zip->addEmptyDir($zip_path);
                        mohtavanegar_add_files_to_zip($zip, $file_path, $zip_path);
                    }
                }
            }
            closedir($handle);
        }
    }
}
