<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Theme Settings Import/Export Functions
 */

/**
 * Export theme settings
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_theme_settings($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create settings directory
    $settings_dir = $export_dir . '/theme-settings';
    if (!is_dir($settings_dir)) {
        mkdir($settings_dir, 0755, true);
    }
    
    // Get all theme settings
    $settings = array(
        'basic_settings' => get_option('basic_settings', array()),
        'header_template' => get_option('header_template', ''),
        'header_type' => get_option('header_type', 'default'),
        'footer_template' => get_option('footer_template', ''),
        'footer_type' => get_option('footer_type', 'default'),
        'single_post_template' => get_option('single_post_template', ''),
        'single_product_template' => get_option('single_product_template', ''),
        'typography_settings' => get_option('typography_settings', array()),
        'site_logo' => get_option('site_logo', '')
    );
    
    // If site logo exists, get the URL for reference
    if (!empty($settings['site_logo'])) {
        $logo_url = wp_get_attachment_url($settings['site_logo']);
        if ($logo_url) {
            $settings['site_logo_url'] = $logo_url;
        }
    }
    
    // Export customizer settings
    $theme_mods = get_theme_mods();
    if ($theme_mods) {
        $settings['theme_mods'] = $theme_mods;
    }
    
    // Save settings to file
    $settings_file = $settings_dir . '/settings.json';
    file_put_contents($settings_file, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($settings);
    $result['message'] = __('تنظیمات قالب با موفقیت برون‌ریزی شدند.', 'mohtavanegar');
    
    return $result;
}

/**
 * Import theme settings
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_theme_settings($import_dir) {
    // Start logging for debugging
    error_log('========== THEME SETTINGS IMPORT START ==========');
    error_log('Function mohtavanegar_import_theme_settings called with directory: ' . $import_dir);
    
    // Initialize result
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Check if import directory exists
    if (!is_dir($import_dir)) {
        error_log('Import directory does not exist: ' . $import_dir);
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری اصلی درون‌ریزی یافت نشد.', 'mohtavanegar');
        error_log('========== THEME SETTINGS IMPORT END ==========');
        return $result;
    }
    
    // Log directory contents for debugging
    $dir_contents = scandir($import_dir);
    error_log('Import directory contents: ' . print_r($dir_contents, true));
    
    // Look for theme-settings directory
    $settings_dir = $import_dir . '/theme-settings';
    error_log('Looking for settings directory: ' . $settings_dir);
    
    if (!is_dir($settings_dir)) {
        error_log('Settings directory not found: ' . $settings_dir);
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری تنظیمات قالب یافت نشد.', 'mohtavanegar');
        error_log('========== THEME SETTINGS IMPORT END ==========');
        return $result;
    }
    
    // Look for settings.json file
    $settings_file = $settings_dir . '/settings.json';
    error_log('Looking for settings file: ' . $settings_file);
    
    if (!file_exists($settings_file)) {
        error_log('Settings file not found: ' . $settings_file);
        $result['status'] = 'error';
        $result['message'] = __('فایل تنظیمات یافت نشد.', 'mohtavanegar');
        error_log('========== THEME SETTINGS IMPORT END ==========');
        return $result;
    }
    
    // Read and decode settings file
    $file_contents = @file_get_contents($settings_file);
    if ($file_contents === false) {
        error_log('Error reading settings file: ' . error_get_last()['message']);
        $result['status'] = 'error';
        $result['message'] = __('خطا در خواندن فایل تنظیمات.', 'mohtavanegar');
        error_log('========== THEME SETTINGS IMPORT END ==========');
        return $result;
    }
    
    $settings = json_decode($file_contents, true);
    if (!$settings) {
        error_log('Error decoding settings JSON: ' . json_last_error_msg());
        $result['status'] = 'error';
        $result['message'] = __('فایل تنظیمات نامعتبر است.', 'mohtavanegar');
        error_log('========== THEME SETTINGS IMPORT END ==========');
        return $result;
    }
    
    error_log('Settings successfully decoded, found ' . count($settings) . ' settings');
    error_log('Setting keys: ' . implode(', ', array_keys($settings)));
    
    try {
        // Update basic settings
        if (isset($settings['basic_settings'])) {
            error_log('Updating basic_settings');
            update_option('basic_settings', $settings['basic_settings']);
        }
        
        // Update header settings
        if (isset($settings['header_template'])) {
            error_log('Updating header_template');
            update_option('header_template', $settings['header_template']);
        }
        
        if (isset($settings['header_type'])) {
            error_log('Updating header_type');
            update_option('header_type', $settings['header_type']);
        }
        
        // Update footer settings
        if (isset($settings['footer_template'])) {
            error_log('Updating footer_template');
            update_option('footer_template', $settings['footer_template']);
        }
        
        if (isset($settings['footer_type'])) {
            error_log('Updating footer_type');
            update_option('footer_type', $settings['footer_type']);
        }
        
        // Update template settings
        if (isset($settings['single_post_template'])) {
            error_log('Updating single_post_template');
            update_option('single_post_template', $settings['single_post_template']);
        }
        
        if (isset($settings['single_product_template'])) {
            error_log('Updating single_product_template');
            update_option('single_product_template', $settings['single_product_template']);
        }
        
        // Update typography settings
        if (isset($settings['typography_settings'])) {
            error_log('Updating typography_settings');
            update_option('typography_settings', $settings['typography_settings']);
        }
        
        // Update site logo
        if (isset($settings['site_logo_url']) && !empty($settings['site_logo_url'])) {
            error_log('Processing site logo, URL: ' . $settings['site_logo_url']);
            $attachment_id = mohtavanegar_get_attachment_id_by_original_url($settings['site_logo_url']);
            
            if ($attachment_id) {
                error_log('Found attachment ID: ' . $attachment_id);
                update_option('site_logo', $attachment_id);
            } else {
                error_log('Could not find attachment ID for logo');
            }
        }
        
        // Import theme mods
        if (isset($settings['theme_mods']) && is_array($settings['theme_mods'])) {
            error_log('Importing theme mods, found ' . count($settings['theme_mods']) . ' items');
            foreach ($settings['theme_mods'] as $key => $value) {
                // Skip nav_menu_locations as they are handled by the menu importer
                if ($key === 'nav_menu_locations') {
                    error_log('Skipping nav_menu_locations');
                    continue;
                }
                
                error_log('Setting theme_mod: ' . $key);
                set_theme_mod($key, $value);
            }
        }
        
        // Clear Elementor cache if exists
        if (function_exists('mohtavanegar_clear_elementor_cache')) {
            error_log('Clearing Elementor cache');
            mohtavanegar_clear_elementor_cache();
        }
        
        $result['count'] = count($settings);
        $result['message'] = __('تنظیمات قالب با موفقیت درون‌ریزی شدند.', 'mohtavanegar');
        
    } catch (Exception $e) {
        error_log('Exception during import: ' . $e->getMessage());
        $result['status'] = 'error';
        $result['message'] = __('خطا در درون‌ریزی تنظیمات: ', 'mohtavanegar') . $e->getMessage();
    } catch (Error $e) {
        error_log('PHP Error during import: ' . $e->getMessage());
        $result['status'] = 'error';
        $result['message'] = __('خطای PHP در درون‌ریزی تنظیمات: ', 'mohtavanegar') . $e->getMessage();
    }
    
    error_log('Import completed with status: ' . $result['status']);
    error_log('========== THEME SETTINGS IMPORT END ==========');
    
    return $result;
}
