<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Media Import/Export Functions
 */

/**
 * Export media files
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_media($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create media directory
    $media_dir = $export_dir . '/media';
    if (!is_dir($media_dir)) {
        mkdir($media_dir, 0755, true);
    }
    
    // Get all attachments
    $attachments = get_posts(array(
        'post_type' => 'attachment',
        'post_status' => 'inherit',
        'posts_per_page' => -1
    ));
    
    $attachments_data = array();
    $copied_files = 0;
    
    foreach ($attachments as $attachment) {
        $file_path = get_attached_file($attachment->ID);
        
        if (!file_exists($file_path)) {
            continue;
        }
        
        // Get file info
        $file_info = pathinfo($file_path);
        $file_name = $file_info['basename'];
        
        // Copy file to export directory
        $destination = $media_dir . '/' . $file_name;
        if (copy($file_path, $destination)) {
            $copied_files++;
            
            // Get attachment meta
            $attachment_meta = wp_get_attachment_metadata($attachment->ID);
            
            // Get attachment URL
            $attachment_url = wp_get_attachment_url($attachment->ID);
            
            $attachments_data[] = array(
                'ID' => $attachment->ID,
                'post_title' => $attachment->post_title,
                'post_name' => $attachment->post_name,
                'post_date' => $attachment->post_date,
                'post_mime_type' => $attachment->post_mime_type,
                'guid' => $attachment->guid,
                'file_name' => $file_name,
                'file_path' => $file_path,
                'meta' => $attachment_meta,
                'url' => $attachment_url,
                'alt' => get_post_meta($attachment->ID, '_wp_attachment_image_alt', true)
            );
        }
    }
    
    // Save attachments data
    $attachments_file = $media_dir . '/attachments.json';
    file_put_contents($attachments_file, json_encode($attachments_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = $copied_files;
    $result['message'] = sprintf(__('%d فایل رسانه با موفقیت برون‌ریزی شدند.', 'mohtavanegar'), $copied_files);
    
    return $result;
}

/**
 * Import media files
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_media($import_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    $media_dir = $import_dir . '/media';
    
    // Check if media directory exists
    if (!is_dir($media_dir)) {
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری رسانه‌ها یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // Import attachments
    $attachments_file = $media_dir . '/attachments.json';
    if (file_exists($attachments_file)) {
        $attachments_data = json_decode(file_get_contents($attachments_file), true);
        
        if ($attachments_data) {
            $imported_count = 0;
            $attachment_id_mapping = array();
            
            // Get WordPress upload directory
            $upload_dir = wp_upload_dir();
            
            foreach ($attachments_data as $attachment_data) {
                // Check if file exists in the import directory
                $source_file = $media_dir . '/' . $attachment_data['file_name'];
                
                if (!file_exists($source_file)) {
                    continue;
                }
                
                // Check if attachment already exists by file name
                $existing_attachment = mohtavanegar_get_attachment_by_filename($attachment_data['file_name']);
                
                if ($existing_attachment) {
                    // Update existing attachment
                    $attachment_id = $existing_attachment->ID;
                    
                    wp_update_post(array(
                        'ID' => $attachment_id,
                        'post_title' => $attachment_data['post_title']
                    ));
                    
                    // Update alt text
                    if (isset($attachment_data['alt'])) {
                        update_post_meta($attachment_id, '_wp_attachment_image_alt', $attachment_data['alt']);
                    }
                    
                    $attachment_id_mapping[$attachment_data['ID']] = $attachment_id;
                    $attachment_id_mapping[$attachment_data['url']] = wp_get_attachment_url($attachment_id);
                    
                    // Store original URL for reference
                    update_post_meta($attachment_id, '_mohtavanegar_original_url', $attachment_data['url']);
                    
                    $imported_count++;
                } else {
                    // Prepare file destination
                    $year_month = date('Y/m', strtotime($attachment_data['post_date']));
                    $destination_dir = $upload_dir['basedir'] . '/' . $year_month;
                    
                    // Create directory if it doesn't exist
                    if (!is_dir($destination_dir)) {
                        wp_mkdir_p($destination_dir);
                    }
                    
                    $destination_file = $destination_dir . '/' . $attachment_data['file_name'];
                    
                    // Copy file to uploads directory
                    if (copy($source_file, $destination_file)) {
                        // Prepare attachment data
                        $file_type = wp_check_filetype($attachment_data['file_name'], null);
                        
                        $attachment = array(
                            'guid' => $upload_dir['baseurl'] . '/' . $year_month . '/' . $attachment_data['file_name'],
                            'post_mime_type' => $file_type['type'] ? $file_type['type'] : $attachment_data['post_mime_type'],
                            'post_title' => $attachment_data['post_title'],
                            'post_content' => '',
                            'post_status' => 'inherit'
                        );
                        
                        // Insert attachment
                        $attachment_id = wp_insert_attachment($attachment, $destination_file);
                        
                        if (!is_wp_error($attachment_id)) {
                            // Generate attachment metadata
                            require_once(ABSPATH . 'wp-admin/includes/image.php');
                            $attachment_data_db = wp_generate_attachment_metadata($attachment_id, $destination_file);
                            wp_update_attachment_metadata($attachment_id, $attachment_data_db);
                            
                            // Update alt text
                            if (isset($attachment_data['alt'])) {
                                update_post_meta($attachment_id, '_wp_attachment_image_alt', $attachment_data['alt']);
                            }
                            
                            $attachment_id_mapping[$attachment_data['ID']] = $attachment_id;
                            $attachment_id_mapping[$attachment_data['url']] = wp_get_attachment_url($attachment_id);
                            
                            // Store original URL for reference
                            update_post_meta($attachment_id, '_mohtavanegar_original_url', $attachment_data['url']);
                            
                            $imported_count++;
                        }
                    }
                }
            }
            
            // Store attachment ID mapping for other importers to use
            update_option('mohtavanegar_attachment_id_mapping', $attachment_id_mapping, false);
            
            $result['count'] = $imported_count;
            $result['message'] = sprintf(__('%d فایل رسانه با موفقیت درون‌ریزی شدند.', 'mohtavanegar'), $imported_count);
        }
    }
    
    return $result;
}

/**
 * Get attachment by filename
 *
 * @param string $filename Filename to search for
 * @return WP_Post|false Attachment post or false if not found
 */
function mohtavanegar_get_attachment_by_filename($filename) {
    $args = array(
        'post_type' => 'attachment',
        'post_status' => 'inherit',
        'posts_per_page' => 1,
        'meta_query' => array(
            array(
                'key' => '_wp_attached_file',
                'value' => $filename,
                'compare' => 'LIKE'
            )
        )
    );
    
    $attachments = get_posts($args);
    
    if (!empty($attachments)) {
        return $attachments[0];
    }
    
    return false;
}
