<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Pages Import/Export Functions
 */

/**
 * Export pages
 *
 * @param string $export_dir Directory to export to
 * @return array Status information
 */
function mohtavanegar_export_pages($export_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    // Create pages directory
    $pages_dir = $export_dir . '/pages';
    if (!is_dir($pages_dir)) {
        mkdir($pages_dir, 0755, true);
    }
    
    // Export pages
    $pages = get_pages(array(
        'post_status' => 'publish',
        'sort_column' => 'menu_order',
        'sort_order' => 'ASC'
    ));
    
    $pages_data = array();
    foreach ($pages as $page) {
        // Get page meta
        $page_meta = get_post_meta($page->ID);
        $filtered_meta = array();
        
        // Filter out unnecessary meta
        foreach ($page_meta as $meta_key => $meta_value) {
            // Keep all Elementor-related meta keys and other important meta
            if (substr($meta_key, 0, 1) === '_' && 
                (strpos($meta_key, '_elementor_') === 0 || 
                 in_array($meta_key, array(
                     '_thumbnail_id',
                     '_wp_page_template',
                     '_wp_page_template_conditions',
                     '_wp_page_template_conditions_priority'
                 )))) {
                $filtered_meta[$meta_key] = maybe_unserialize($meta_value[0]);
            }
        }
        
        // Check if this is an Elementor page
        $is_elementor_page = false;
        if (isset($page_meta['_elementor_edit_mode']) && $page_meta['_elementor_edit_mode'][0] === 'builder') {
            $is_elementor_page = true;
        }
        
        // Get featured image
        $featured_image_id = get_post_thumbnail_id($page->ID);
        $featured_image_url = '';
        $featured_image_meta = array();
        
        if ($featured_image_id) {
            $featured_image_url = wp_get_attachment_url($featured_image_id);
            $featured_image_meta = wp_get_attachment_metadata($featured_image_id);
        }
        
        // Get page template and conditions
        $page_template = get_page_template_slug($page->ID);
        $page_template_conditions = get_post_meta($page->ID, '_wp_page_template_conditions', true);
        
        // Get page parent info
        $parent_info = array();
        if ($page->post_parent) {
            $parent_page = get_post($page->post_parent);
            if ($parent_page) {
                $parent_info = array(
                    'ID' => $parent_page->ID,
                    'post_name' => $parent_page->post_name,
                    'post_title' => $parent_page->post_title
                );
            }
        }
        
        $pages_data[] = array(
            'ID' => $page->ID,
            'post_title' => $page->post_title,
            'post_content' => $page->post_content,
            'post_excerpt' => $page->post_excerpt,
            'post_date' => $page->post_date,
            'post_name' => $page->post_name,
            'post_status' => $page->post_status,
            'post_author' => $page->post_author,
            'menu_order' => $page->menu_order,
            'post_parent' => $page->post_parent,
            'parent_info' => $parent_info,
            'meta' => $filtered_meta,
            'featured_image' => array(
                'url' => $featured_image_url,
                'meta' => $featured_image_meta
            ),
            'page_template' => $page_template,
            'page_template_conditions' => $page_template_conditions,
            'is_elementor_page' => $is_elementor_page
        );
        
        // Log Elementor page export
        if ($is_elementor_page) {
            error_log('Exporting Elementor page: ' . $page->post_title . ' (ID: ' . $page->ID . ')');
        }
    }
    
    $pages_file = $pages_dir . '/pages.json';
    file_put_contents($pages_file, json_encode($pages_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    $result['count'] = count($pages);
    $result['message'] = sprintf(__('%d برگه با موفقیت برون‌ریزی شدند.', 'mohtavanegar'), count($pages));
    
    return $result;
}

/**
 * Import pages
 *
 * @param string $import_dir Directory to import from
 * @return array Status information
 */
function mohtavanegar_import_pages($import_dir) {
    $result = array(
        'status' => 'success',
        'message' => '',
        'count' => 0
    );
    
    $pages_dir = $import_dir . '/pages';
    
    // Check if pages directory exists
    if (!is_dir($pages_dir)) {
        $result['status'] = 'error';
        $result['message'] = __('دایرکتوری برگه‌ها یافت نشد.', 'mohtavanegar');
        return $result;
    }
    
    // Import pages
    $pages_file = $pages_dir . '/pages.json';
    if (file_exists($pages_file)) {
        $pages_data = json_decode(file_get_contents($pages_file), true);
        
        if ($pages_data) {
            $imported_count = 0;
            $page_id_mapping = array();
            
            // First pass: Create all pages
            foreach ($pages_data as $page_data) {
                // Check if page exists by slug
                $existing_page = get_page_by_path($page_data['post_name'], OBJECT, 'page');
                
                if ($existing_page) {
                    // Update existing page
                    $page_id = $existing_page->ID;
                    
                    wp_update_post(array(
                        'ID' => $page_id,
                        'post_title' => $page_data['post_title'],
                        'post_content' => $page_data['post_content'],
                        'post_excerpt' => $page_data['post_excerpt'],
                        'post_status' => $page_data['post_status'],
                        'menu_order' => $page_data['menu_order']
                    ));
                } else {
                    // Create new page
                    $page_id = wp_insert_post(array(
                        'post_title' => $page_data['post_title'],
                        'post_content' => $page_data['post_content'],
                        'post_excerpt' => $page_data['post_excerpt'],
                        'post_date' => $page_data['post_date'],
                        'post_name' => $page_data['post_name'],
                        'post_status' => $page_data['post_status'],
                        'post_author' => get_current_user_id(),
                        'post_type' => 'page',
                        'menu_order' => $page_data['menu_order']
                    ));
                }
                
                if ($page_id && !is_wp_error($page_id)) {
                    $page_id_mapping[$page_data['ID']] = $page_id;
                }
            }
            
            // Second pass: Update meta, featured image, and process Elementor data
            foreach ($pages_data as $page_data) {
                $page_id = $page_id_mapping[$page_data['ID']] ?? null;
                
                if ($page_id && !is_wp_error($page_id)) {
                    // Set page template
                    if (!empty($page_data['page_template'])) {
                        update_post_meta($page_id, '_wp_page_template', $page_data['page_template']);
                    }
                    
                    // Set page template conditions
                    if (!empty($page_data['page_template_conditions'])) {
                        update_post_meta($page_id, '_wp_page_template_conditions', $page_data['page_template_conditions']);
                    }
                    
                    // Set meta
                    if (!empty($page_data['meta'])) {
                        foreach ($page_data['meta'] as $meta_key => $meta_value) {
                            // Special handling for Elementor data
                            if ($meta_key === '_elementor_data' && !empty($meta_value)) {
                                $elementor_data = json_decode($meta_value, true);
                                if ($elementor_data) {
                                    // Get template ID mapping from the elementor templates import
                                    $template_id_mapping = get_option('mohtavanegar_template_id_mapping', array());
                                    
                                    // Process the Elementor data using our processor function
                                    $elementor_data = mohtavanegar_process_elementor_content_data($elementor_data, $template_id_mapping);
                                    
                                    // Update the meta with processed data
                                    update_post_meta($page_id, $meta_key, wp_slash(json_encode($elementor_data)));
                                    
                                    // Make sure Elementor edit mode is set
                                    update_post_meta($page_id, '_elementor_edit_mode', 'builder');
                                    
                                    // Set Elementor version if not already set
                                    if (class_exists('\Elementor\Plugin')) {
                                        if (defined('ELEMENTOR_VERSION')) {
                                            $elementor_version = ELEMENTOR_VERSION;
                                        } else {
                                            $elementor_version = '3.0.0';
                                        }
                                        update_post_meta($page_id, '_elementor_version', $elementor_version);
                                    }
                                    
                                    // Log the processing
                                    error_log('Processed Elementor data for page ID: ' . $page_id);
                                } else {
                                    update_post_meta($page_id, $meta_key, $meta_value);
                                }
                            } else {
                                update_post_meta($page_id, $meta_key, $meta_value);
                            }
                        }
                    }
                    
                    // Set featured image
                    if (!empty($page_data['featured_image']['url'])) {
                        $attachment_id = mohtavanegar_get_attachment_id_by_url($page_data['featured_image']['url']);
                        if ($attachment_id) {
                            set_post_thumbnail($page_id, $attachment_id);
                            
                            // Update attachment metadata if available
                            if (!empty($page_data['featured_image']['meta'])) {
                                wp_update_attachment_metadata($attachment_id, $page_data['featured_image']['meta']);
                            }
                        }
                    }
                    
                    // Update parent page if needed
                    if (!empty($page_data['post_parent']) && isset($page_id_mapping[$page_data['post_parent']])) {
                        wp_update_post(array(
                            'ID' => $page_id,
                            'post_parent' => $page_id_mapping[$page_data['post_parent']]
                        ));
                    }
                    
                    $imported_count++;
                }
            }
            
            $result['count'] = $imported_count;
            $result['message'] = sprintf(__('%d برگه با موفقیت درون‌ریزی شدند.', 'mohtavanegar'), $imported_count);
        }
    }
    
    return $result;
}
