<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Plugin Installer Page
 * Allows for easy installation of recommended plugins from the WordPress repository
 */

// Add Plugin Installer submenu to Theme Settings menu
function mohtavanegar_add_plugin_installer_submenu() {
    add_submenu_page(
        'theme-settings',           // Parent slug (theme-settings is the main menu)
        'نصب پلاگین ها',             // Page title
        'نصب پلاگین ها',             // Menu title
        'install_plugins',          // Capability
        'plugin-installer',         // Menu slug
        'mohtavanegar_plugin_installer_page' // Callback function
    );
}
add_action('admin_menu', 'mohtavanegar_add_plugin_installer_submenu', 11); // Priority 11 to ensure it runs after the main menu

// Plugin Installer Page Content
function mohtavanegar_plugin_installer_page() {
    if (!current_user_can('install_plugins')) {
        wp_die(__('شما دسترسی لازم برای نصب پلاگین‌ها را ندارید.', 'mohtavanegar'));
    }
    
    // Define recommended plugins
    $recommended_plugins = array(
        'elementor' => array(
            'name' => 'Elementor',
            'slug' => 'elementor',
            'description' => 'نسخه حرفه‌ای صفحه‌ساز المنتور با امکانات پیشرفته',
            'icon' => 'https://ps.w.org/elementor/assets/icon-128x128.gif'
        ),
        'elementor-pro' => array(
            'name' => 'Elementor Pro',
            'slug' => 'elementor-pro',
            'description' => 'نسخه حرفه‌ای صفحه‌ساز المنتور با امکانات پیشرفته',
            'icon' => 'https://ps.w.org/elementor/assets/icon-128x128.gif',
            'source' => 'https://medpress.net/plugin-need/download/elementor-pro' // Custom source for manual download
        ),
        'revslider' => array(
            'name' => 'Slider Revolution',
            'slug' => 'revslider',
            'description' => 'اسلایدر حرفه‌ای برای ایجاد اسلایدشوهای جذاب',
            'icon' => 'https://www.sliderrevolution.com/wp-content/uploads/2020/02/srlogo.png',
            'source' => 'https://medpress.net/plugin-need/download/slider-revolution'
        ),
        'litespeed-cache' => array(
            'name' => 'LiteSpeed Cache',
            'slug' => 'litespeed-cache',
            'description' => 'افزونه کش LiteSpeed برای بهبود سرعت بارگذاری سایت',
            'icon' => 'https://ps.w.org/litespeed-cache/assets/icon-128x128.png'
        ),
        'classic-editor' => array(
            'name' => 'Classic Editor',
            'slug' => 'classic-editor',
            'description' => 'بازگرداندن ویرایشگر کلاسیک وردپرس',
            'icon' => 'https://ps.w.org/classic-editor/assets/icon-128x128.png'
        )
    );
    
    // Process individual plugin installation if requested
    if (isset($_POST['install_plugin']) && !empty($_POST['plugin_slug'])) {
        // Verify nonce
        check_admin_referer('mohtavanegar_install_plugin_' . $_POST['plugin_slug']);
        
        $plugin_slug = sanitize_text_field($_POST['plugin_slug']);
        
        // Include necessary files for plugin installation
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-ajax-upgrader-skin.php';
        
        $status = mohtavanegar_get_plugin_status($plugin_slug);
        
        if ($status === 'not_installed') {
            // Install the plugin
            $install_results = mohtavanegar_install_plugin($plugin_slug);
        } elseif ($status === 'inactive') {
            // Activate the plugin
            $install_results = mohtavanegar_activate_plugin($plugin_slug);
        } else {
            $install_results = array(
                'success' => true,
                'message' => sprintf(__('پلاگین %s در حال حاضر فعال است.', 'mohtavanegar'), $recommended_plugins[$plugin_slug]['name'])
            );
        }
    }
    
    // Check plugin statuses
    foreach ($recommended_plugins as $key => $plugin) {
        $recommended_plugins[$key]['status'] = mohtavanegar_get_plugin_status($plugin['slug']);
    }
    
    ?>
    <div class="plugin-installer-wrap">
        <h2><?php _e('نصب پلاگین‌های پیشنهادی', 'mohtavanegar'); ?></h2>
        <p><?php _e('پلاگین‌های زیر برای عملکرد بهتر سایت پیشنهاد می‌شوند.', 'mohtavanegar'); ?></p>
        
        <?php if (isset($install_results)): ?>
        <div class="notice notice-info">
            <h3><?php _e('نتیجه نصب:', 'mohtavanegar'); ?></h3>
            <p>
                <strong><?php echo $recommended_plugins[$_POST['plugin_slug']]['name']; ?>:</strong> 
                <?php echo $install_results['message']; ?>
            </p>
        </div>
        <?php endif; ?>
    
        <div class="plugin-cards">
            <?php foreach ($recommended_plugins as $key => $plugin) : ?>
                <div class="plugin-card">
                    <div class="plugin-card-top">
                        <div class="plugin-need-icon">
                            <img src="<?php echo esc_url($plugin['icon']); ?>" alt="<?php echo esc_attr($plugin['name']); ?>">
                        </div>
                        <div class="plugin-info">
                            <h3 class="plugin-name"><?php echo esc_html($plugin['name']); ?></h3>
                            <p class="plugin-description"><?php echo esc_html($plugin['description']); ?></p>
                        </div>
                    </div>
                    <div class="plugin-card-bottom">
                        <?php if ($plugin['status'] === 'active') : ?>
                            <span class="plugin-status active"><?php _e('فعال', 'mohtavanegar'); ?></span>
                        <?php elseif ($plugin['status'] === 'inactive') : ?>
                            <form method="post" action="">
                                <?php wp_nonce_field('mohtavanegar_install_plugin_' . $key); ?>
                                <input type="hidden" name="plugin_slug" value="<?php echo esc_attr($key); ?>">
                                <input type="hidden" name="install_plugin" value="1">
                                <button type="submit" class="button activate-plugin"><?php _e('فعال‌سازی', 'mohtavanegar'); ?></button>
                            </form>
                        <?php else : ?>
                            <form method="post" action="">
                                <?php wp_nonce_field('mohtavanegar_install_plugin_' . $key); ?>
                                <input type="hidden" name="plugin_slug" value="<?php echo esc_attr($key); ?>">
                                <input type="hidden" name="install_plugin" value="1">
                                <button type="submit" class="button install-plugin"><?php _e('نصب', 'mohtavanegar'); ?></button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        
    </div>
    <?php
}

/**
 * Get plugin status (not installed, inactive, active)
 */
function mohtavanegar_get_plugin_status($plugin_slug) {
    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    
    $installed_plugins = get_plugins();
    
    // Check for plugin by slug
    foreach ($installed_plugins as $plugin_path => $plugin_data) {
        if (strpos($plugin_path, $plugin_slug . '/') === 0) {
            return is_plugin_active($plugin_path) ? 'active' : 'inactive';
        }
    }
    
    return 'not_installed';
}

/**
 * Helper function to install a plugin
 */
function mohtavanegar_install_plugin($plugin_slug) {
    // Include necessary files for plugin installation if not already included
    if (!function_exists('plugins_api')) {
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
    }
    if (!class_exists('Plugin_Upgrader')) {
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
    }
    if (!class_exists('WP_Ajax_Upgrader_Skin')) {
        require_once ABSPATH . 'wp-admin/includes/class-wp-ajax-upgrader-skin.php';
    }
    
    // Get plugin info from WordPress.org
    $api = plugins_api('plugin_information', array(
        'slug' => $plugin_slug,
        'fields' => array(
            'short_description' => false,
            'sections' => false,
            'requires' => false,
            'rating' => false,
            'ratings' => false,
            'downloaded' => false,
            'last_updated' => false,
            'added' => false,
            'tags' => false,
            'compatibility' => false,
            'homepage' => false,
            'donate_link' => false,
        ),
    ));
    
    if (is_wp_error($api)) {
        return array(
            'success' => false,
            'message' => $api->get_error_message()
        );
    }
    
    // Install the plugin
    $upgrader = new Plugin_Upgrader(new WP_Ajax_Upgrader_Skin());
    $install = $upgrader->install($api->download_link);
    
    if (is_wp_error($install)) {
        return array(
            'success' => false,
            'message' => $install->get_error_message()
        );
    }
    
    if (false === $install) {
        return array(
            'success' => false,
            'message' => __('خطا در نصب پلاگین.', 'mohtavanegar')
        );
    }
    
    // Get the plugin basename
    $plugin_basename = $upgrader->plugin_info();
    
    // Activate the plugin
    $activate_result = mohtavanegar_activate_plugin_by_basename($plugin_basename);
    
    if ($activate_result['success']) {
        return array(
            'success' => true,
            'message' => __('پلاگین با موفقیت نصب و فعال شد.', 'mohtavanegar')
        );
    } else {
        return array(
            'success' => true,
            'message' => __('پلاگین با موفقیت نصب شد اما فعال‌سازی آن با خطا مواجه شد.', 'mohtavanegar')
        );
    }
}

/**
 * Helper function to activate a plugin by slug
 */
function mohtavanegar_activate_plugin($plugin_slug) {
    // Get plugin file path from slug
    $plugin_file = mohtavanegar_get_plugin_file_from_slug($plugin_slug);
    
    if (!$plugin_file) {
        return array(
            'success' => false,
            'message' => sprintf(__('پلاگین %s پیدا نشد.', 'mohtavanegar'), $plugin_slug)
        );
    }
    
    return mohtavanegar_activate_plugin_by_basename($plugin_file);
}

/**
 * Helper function to activate a plugin by its basename
 */
function mohtavanegar_activate_plugin_by_basename($plugin_basename) {
    if (!function_exists('activate_plugin')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    
    $result = activate_plugin($plugin_basename);
    
    if (is_wp_error($result)) {
        return array(
            'success' => false,
            'message' => $result->get_error_message()
        );
    }
    
    return array(
        'success' => true,
        'message' => __('پلاگین با موفقیت فعال شد.', 'mohtavanegar')
    );
}

/**
 * Get plugin file path from slug
 */
function mohtavanegar_get_plugin_file_from_slug($plugin_slug) {
    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    
    $plugins = get_plugins();
    
    foreach ($plugins as $plugin_file => $plugin_info) {
        if (strpos($plugin_file, $plugin_slug . '/') === 0) {
            return $plugin_file;
        }
    }
    
    return false;
}

/**
 * AJAX handler for plugin installation
 */
function mohtavanegar_install_plugin_ajax() {
    // Check nonce
    if (!isset($_POST['security']) || !wp_verify_nonce($_POST['security'], 'mohtavanegar_install_plugin')) {
        wp_send_json_error(array('message' => __('خطای امنیتی. لطفاً صفحه را رفرش کنید.', 'mohtavanegar')));
    }
    
    // Check user capabilities
    if (!current_user_can('install_plugins')) {
        wp_send_json_error(array('message' => __('شما دسترسی لازم برای نصب پلاگین‌ها را ندارید.', 'mohtavanegar')));
    }
    
    // Get plugin slug
    $plugin_slug = isset($_POST['slug']) ? sanitize_text_field($_POST['slug']) : '';
    if (empty($plugin_slug)) {
        wp_send_json_error(array('message' => __('پلاگین مشخص نشده است.', 'mohtavanegar')));
    }
    
    $result = mohtavanegar_install_plugin($plugin_slug);
    
    if ($result['success']) {
        wp_send_json_success(array(
            'message' => $result['message'],
            'activateUrl' => admin_url('admin.php?page=plugin-installer')
        ));
    } else {
        wp_send_json_error(array('message' => $result['message']));
    }
}
add_action('wp_ajax_mohtavanegar_install_plugin', 'mohtavanegar_install_plugin_ajax');

/**
 * Add CSS styles for the plugin installer page
 */
function mohtavanegar_plugin_installer_styles() {
    ?>
    <style>
        .plugin-cards {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-top: 20px;
        }
        
        .plugin-card {
            flex: 0 0 calc(50% - 10px);
            max-width: calc(50% - 10px);
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 5px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .plugin-card-top {
            padding: 20px;
            display: flex;
            align-items: center;
        }
        
        .plugin-need-icon {
            flex: 0 0 60px;
            margin-right: 15px;
        }
        
        .plugin-need-icon img {
            width: 60px;
            height: 60px;
            border-radius: 3px;
        }
        
        .plugin-info {
            flex: 1;
        }
        
        .plugin-name {
            margin: 0 0 10px;
            font-size: 16px;
        }
        
        .plugin-description {
            margin: 0;
            color: #666;
        }
        
        .plugin-card-bottom {
            background: #f7f7f7;
            border-top: 1px solid #ddd;
            padding: 12px 20px;
            display: flex;
            justify-content: flex-end;
        }
        
        .plugin-status {
            display: inline-block;
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 13px;
        }
        
        .plugin-status.active {
            background: #dff0d8;
            color: #3c763d;
        }
        
        @media screen and (max-width: 782px) {
            .plugin-card {
                flex: 0 0 100%;
                max-width: 100%;
            }
        }
    </style>
    <?php
}
