<?php
if (!defined('ABSPATH')) {
    exit;
}

// Add meta box for single post/page header and footer settings
function mohtavanegar_add_single_post_settings_meta_box() {
    $screens = array('post', 'page');
    
    foreach ($screens as $screen) {
        add_meta_box(
            'mohtavanegar_single_post_settings',
            'تنظیمات هدر و فوتر',
            'mohtavanegar_render_single_post_settings_meta_box',
            $screen,
            'side',
            'default'
        );
    }
}
add_action('add_meta_boxes', 'mohtavanegar_add_single_post_settings_meta_box');

// Render meta box content
function mohtavanegar_render_single_post_settings_meta_box($post) {
    // Add nonce for security and authentication
    wp_nonce_field('mohtavanegar_single_post_settings_nonce', 'mohtavanegar_single_post_settings_nonce');
    
    // Get saved values
    $header_template = get_post_meta($post->ID, '_mohtavanegar_custom_header_template', true);
    $footer_template = get_post_meta($post->ID, '_mohtavanegar_custom_footer_template', true);
    $hide_header = get_post_meta($post->ID, '_mohtavanegar_hide_header', true);
    $hide_footer = get_post_meta($post->ID, '_mohtavanegar_hide_footer', true);
    
    // Get all Elementor templates
    $templates = get_posts(array(
        'post_type' => 'elementor_library',
        'posts_per_page' => -1,
        'post_status' => 'publish',
        'meta_query' => array(
            array(
                'key' => '_elementor_template_type',
                'value' => array('header', 'footer', 'page'),
                'compare' => 'IN',
            ),
        ),
    ));
    
    // Separate headers and footers
    $headers = array();
    $footers = array();
    
    foreach ($templates as $template) {
        $template_type = get_post_meta($template->ID, '_elementor_template_type', true);
        if ($template_type === 'header') {
            $headers[] = $template;
        } elseif ($template_type === 'footer') {
            $footers[] = $template;
        } elseif ($template_type === 'page') {
            // Check if it's a header or footer by name (fallback)
            $title = strtolower($template->post_title);
            if (strpos($title, 'header') !== false) {
                $headers[] = $template;
            } elseif (strpos($title, 'footer') !== false) {
                $footers[] = $template;
            }
        }
    }
    ?>
    <div class="mohtavanegar-single-settings">
        <p>
            <label for="mohtavanegar_custom_header_template">
                <strong>قالب هدر:</strong>
            </label>
            <select name="mohtavanegar_custom_header_template" id="mohtavanegar_custom_header_template" class="widefat" style="margin-top: 5px;">
                <option value="">پیش‌فرض</option>
                <?php foreach ($headers as $template) : ?>
                    <option value="<?php echo esc_attr($template->ID); ?>" <?php selected($header_template, $template->ID); ?>>
                        <?php echo esc_html($template->post_title); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </p>
        
        <p>
            <label>
                <input type="checkbox" name="mohtavanegar_hide_header" id="mohtavanegar_hide_header" value="1" <?php checked($hide_header, '1'); ?>>
                <strong>عدم نمایش هدر</strong>
            </label>
            <small class="description" style="display: block; margin-top: 5px; color: #666;">در صورت فعال‌سازی، هدر در این صفحه نمایش داده نخواهد شد.</small>
        </p>
        
        <p>
            <label for="mohtavanegar_custom_footer_template">
                <strong>قالب فوتر:</strong>
            </label>
            <select name="mohtavanegar_custom_footer_template" id="mohtavanegar_custom_footer_template" class="widefat" style="margin-top: 5px;">
                <option value="">پیش‌فرض</option>
                <?php foreach ($footers as $template) : ?>
                    <option value="<?php echo esc_attr($template->ID); ?>" <?php selected($footer_template, $template->ID); ?>>
                        <?php echo esc_html($template->post_title); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </p>
        
        <p>
            <label>
                <input type="checkbox" name="mohtavanegar_hide_footer" id="mohtavanegar_hide_footer" value="1" <?php checked($hide_footer, '1'); ?>>
                <strong>عدم نمایش فوتر</strong>
            </label>
            <small class="description" style="display: block; margin-top: 5px; color: #666;">در صورت فعال‌سازی، فوتر در این صفحه نمایش داده نخواهد شد.</small>
        </p>
    </div>
    <?php
}

// Save meta box data
function mohtavanegar_save_single_post_settings($post_id) {
    // Check if nonce is set
    if (!isset($_POST['mohtavanegar_single_post_settings_nonce'])) {
        return;
    }
    
    // Verify nonce
    if (!wp_verify_nonce($_POST['mohtavanegar_single_post_settings_nonce'], 'mohtavanegar_single_post_settings_nonce')) {
        return;
    }
    
    // If this is an autosave, our form has not been submitted, so we don't want to do anything
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    // Check user permissions
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }
    
    // Save header template
    if (isset($_POST['mohtavanegar_custom_header_template'])) {
        update_post_meta($post_id, '_mohtavanegar_custom_header_template', sanitize_text_field($_POST['mohtavanegar_custom_header_template']));
    }
    
    // Save footer template
    if (isset($_POST['mohtavanegar_custom_footer_template'])) {
        update_post_meta($post_id, '_mohtavanegar_custom_footer_template', sanitize_text_field($_POST['mohtavanegar_custom_footer_template']));
    }
    
    // Save hide header option
    if (isset($_POST['mohtavanegar_hide_header'])) {
        update_post_meta($post_id, '_mohtavanegar_hide_header', '1');
    } else {
        delete_post_meta($post_id, '_mohtavanegar_hide_header');
    }
    
    // Save hide footer option
    if (isset($_POST['mohtavanegar_hide_footer'])) {
        update_post_meta($post_id, '_mohtavanegar_hide_footer', '1');
    } else {
        delete_post_meta($post_id, '_mohtavanegar_hide_footer');
    }
}
add_action('save_post', 'mohtavanegar_save_single_post_settings');

// Apply custom header and footer templates
function mohtavanegar_apply_custom_templates($template_id, $post_id = null) {
    if (is_singular() && !$post_id) {
        $post_id = get_the_ID();
    }
    
    if (!$post_id) {
        return $template_id;
    }
    
    // Get custom template for the current post type
    $custom_template = get_post_meta($post_id, '_mohtavanegar_custom_' . $template_id . '_template', true);
    
    if ($custom_template) {
        return $custom_template;
    }
    
    return $template_id;
}

// Apply custom header template
function mohtavanegar_apply_custom_header($header_id) {
    return mohtavanegar_apply_custom_templates('header');
}
add_filter('theme_mod_header_template', 'mohtavanegar_apply_custom_header');

// Apply custom footer template
function mohtavanegar_apply_custom_footer($footer_id) {
    return mohtavanegar_apply_custom_templates('footer');
}
add_filter('theme_mod_footer_template', 'mohtavanegar_apply_custom_footer');

// Enqueue admin styles
function mohtavanegar_single_post_settings_styles() {
    $screen = get_current_screen();
    if (in_array($screen->id, array('post', 'page'))) {
        echo '<style>
            .mohtavanegar-single-settings select {
                margin-top: 5px;
            }
            .mohtavanegar-single-settings p {
                margin-bottom: 15px;
            }
        </style>';
    }
}
add_action('admin_head', 'mohtavanegar_single_post_settings_styles');
