<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Aparat_Playlist_Widget extends \Elementor\Widget_Base {
    
    public function get_name() {
        return 'aparat_playlist_widget';
    }

    public function get_title() {
        return __('پلی‌لیست آپارات', 'textdomain');
    }

    public function get_icon() {
        return 'eicon-video-playlist';
    }

    public function get_categories() {
        return ['mohtavanegar'];
    }

    public function get_script_depends() {
        return ['aparat-playlist-widget'];
    }

    public function get_style_depends() {
        return ['aparat-playlist-widget'];
    }

    protected function _register_controls() {
        // Content Tab
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('محتوا', 'textdomain'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'video_link',
            [
                'label' => __('لینک ویدیو آپارات', 'textdomain'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'placeholder' => __('https://www.aparat.com/v/...', 'textdomain'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'video_description',
            [
                'label' => __('توضیحات ویدیو', 'textdomain'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'rows' => 3,
                'placeholder' => __('توضیحات مربوط به این ویدیو...', 'textdomain'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'videos',
            [
                'label' => __('ویدیوهای آپارات', 'textdomain'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'video_link' => 'https://www.aparat.com/v/n25rxn0',
                        'video_description' => __('توضیحات نمونه ویدیو', 'textdomain'),
                    ],
                ],
                'title_field' => '{{{ video_description }}}',
            ]
        );

        $this->add_control(
            'view',
            [
                'label' => __('حالت نمایش', 'textdomain'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'grid',
                'options' => [
                    'grid' => __('شبکه‌ای (ریسپانسیو)', 'textdomain'),
                    'list' => __('لیستی', 'textdomain'),
                ],
                'prefix_class' => 'elementor-view-',
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label' => __('تعداد ستون‌ها', 'textdomain'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '3',
                'tablet_default' => '2',
                'mobile_default' => '1',
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                ],
                'selectors' => [
                    '{{WRAPPER}}.elementor-view-grid .aparat-container' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
                'condition' => [
                    'view' => 'grid',
                ],
            ]
        );

        $this->add_control(
            'show_descriptions',
            [
                'label' => __('نمایش توضیحات', 'textdomain'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('نمایش', 'textdomain'),
                'label_off' => __('مخفی', 'textdomain'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        // Style Tab
        $this->start_controls_section(
            'style_section',
            [
                'label' => __('استایل‌ها', 'textdomain'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'item_style_heading',
            [
                'label' => __('استایل آیتم‌ها', 'textdomain'),
                'type' => \Elementor\Controls_Manager::HEADING,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'item_background',
                'label' => __('پس‌زمینه', 'textdomain'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .aparat-item',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_box_shadow',
                'label' => __('سایه', 'textdomain'),
                'selector' => '{{WRAPPER}} .aparat-item',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'label' => __('حاشیه', 'textdomain'),
                'selector' => '{{WRAPPER}} .aparat-item',
            ]
        );

        $this->add_control(
            'item_border_radius',
            [
                'label' => __('گردی گوشه‌ها', 'textdomain'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .aparat-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => __('فاصله داخلی', 'textdomain'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .aparat-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_margin',
            [
                'label' => __('فاصله خارجی', 'textdomain'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .aparat-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Description Style
        $this->add_control(
            'description_style_heading',
            [
                'label' => __('استایل توضیحات', 'textdomain'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_descriptions' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => __('رنگ متن توضیحات', 'textdomain'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .aparat-description' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'show_descriptions' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'label' => __('تایپوگرافی توضیحات', 'textdomain'),
                'selector' => '{{WRAPPER}} .aparat-description',
                'condition' => [
                    'show_descriptions' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'description_margin_top',
            [
                'label' => __('فاصله توضیحات از بالا', 'textdomain'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .aparat-description' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_descriptions' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Extract video ID from Aparat URL
     */
    private function extract_video_id($url) {
        if (empty($url)) {
            return '';
        }
        
        // Extract video ID from URLs like:
        // https://www.aparat.com/v/n25rxn0
        // https://aparat.com/v/n25rxn0
        // https://www.aparat.com/v/n25rxn0/
        // aparat.com/v/n25rxn0
        
        if (preg_match('/aparat\.com\/v\/([a-zA-Z0-9_-]+)/i', $url, $matches)) {
            return $matches[1];
        }
        
        // Handle cases where only the ID might be provided
        if (preg_match('/^[a-zA-Z0-9_-]{7,15}$/', $url)) {
            return $url;
        }

        return '';
    }

    /**
     * Generate Aparat embed code
     */
    private function generate_embed_code($video_url) {
        $video_id = $this->extract_video_id($video_url);
        
        if (empty($video_id)) {
            return '';
        }
        
        $random_div_id = 'aparat_' . uniqid();
        
        $embed_code = sprintf(
            '<div id="%s"><script type="text/JavaScript" src="https://www.aparat.com/embed/%s?data[rnddiv]=%s&data[responsive]=yes"></script></div>',
            esc_attr($random_div_id),
            esc_attr($video_id),
            esc_attr($random_div_id)
        );
        
        return $embed_code;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="aparat-playlist-widget">
            <div class="aparat-container elementor-view-<?php echo esc_attr($settings['view']); ?>">
                <?php foreach ($settings['videos'] as $index => $item) : 
                    $embed_code = $this->generate_embed_code($item['video_link']);
                    $description = !empty($item['video_description']) ? $item['video_description'] : '';
                    ?>
                    <div class="aparat-item">
                        <?php if (!empty($embed_code)) : ?>
                            <div class="aparat-video-wrapper">
                                <?php echo $embed_code; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($description) && $settings['show_descriptions'] === 'yes') : ?>
                            <div class="aparat-description">
                                <?php echo esc_html($description); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
}


