<?php
namespace Elementor;

if (!defined('ABSPATH')) {
    exit;
}

class Mohtavanegar_Language_Image_Widget extends Widget_Base {

    public function get_name() {
        return 'mohtavanegar-language-image';
    }

    public function get_title() {
        return esc_html__('تصویر بر اساس زبان', 'mohtavanegar');
    }

    public function get_icon() {
        return 'eicon-image';
    }

    public function get_categories() {
        return ['mohtavanegar'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('لیست تصاویر', 'mohtavanegar'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();
        $repeater->add_control(
            'lang_code',
            [
                'label'       => esc_html__('کد زبان (مثال: fa یا en-US)', 'mohtavanegar'),
                'type'        => Controls_Manager::TEXT,
                'default'     => 'fa',
                'placeholder' => 'fa یا en یا en-US',
            ]
        );
        $repeater->add_control(
            'image',
            [
                'label'   => esc_html__('انتخاب تصویر', 'mohtavanegar'),
                'type'    => Controls_Manager::MEDIA,
                'default' => [],
            ]
        );
        $repeater->add_control(
            'alt',
            [
                'label'       => esc_html__('متن جایگزین (alt)', 'mohtavanegar'),
                'type'        => Controls_Manager::TEXT,
                'default'     => '',
                'placeholder' => '',
            ]
        );

        $this->add_control(
            'items',
            [
                'label'       => esc_html__('آیتم‌ها', 'mohtavanegar'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'default'     => [
                    [ 'lang_code' => 'fa', 'alt' => '' ],
                    [ 'lang_code' => 'en', 'alt' => '' ],
                ],
                'title_field' => '{{{ lang_code }}}',
            ]
        );

        $this->add_control(
            'default_image',
            [
                'label'   => esc_html__('تصویر پیش‌فرض (در صورت نبود تطابق)', 'mohtavanegar'),
                'type'    => Controls_Manager::MEDIA,
                'default' => [],
            ]
        );

        $this->add_control(
            'link',
            [
                'label' => esc_html__('لینک سراسری تصویر', 'mohtavanegar'),
                'type'  => Controls_Manager::URL,
                'placeholder' => 'https://example.com/',
                'options' => ['url','is_external','nofollow'],
                'default' => [
                    'url' => '', // در رندر در صورت خالی بودن به home_url برمی‌گردد
                    'is_external' => false,
                    'nofollow' => false,
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('استایل', 'mohtavanegar'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'image_width',
            [
                'label'      => esc_html__('عرض تصویر', 'mohtavanegar'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px', 'vw'],
                'range'      => [
                    '%'  => ['min' => 1, 'max' => 100],
                    'px' => ['min' => 10, 'max' => 2000],
                    'vw' => ['min' => 1, 'max' => 100],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mn-lang-image' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_radius',
            [
                'label'      => esc_html__('حاشیه گرد', 'mohtavanegar'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .mn-lang-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'image_shadow',
                'selector' => '{{WRAPPER}} .mn-lang-image',
            ]
        );

        $this->end_controls_section();
    }

    private function normalize_lang($lang) {
        $lang = strtolower(trim((string)$lang));
        if ($lang === '') return '';
        // Common forms: fa, fa-ir, en, en-us
        return $lang;
    }

    private function base_lang($lang) {
        $parts = explode('-', $lang);
        return $parts[0] ?? $lang;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $items = isset($settings['items']) && is_array($settings['items']) ? $settings['items'] : [];
        $default_image = isset($settings['default_image']['url']) ? $settings['default_image']['url'] : '';
        $link_settings = isset($settings['link']) && is_array($settings['link']) ? $settings['link'] : [];
        $link_url = !empty($link_settings['url']) ? $link_settings['url'] : home_url('/');
        $link_is_external = !empty($link_settings['is_external']);
        $link_nofollow = !empty($link_settings['nofollow']);

        // Build map lang -> image
        $map = [];
        foreach ($items as $item) {
            $code = isset($item['lang_code']) ? $this->normalize_lang($item['lang_code']) : '';
            $url  = isset($item['image']['url']) ? $item['image']['url'] : '';
            $alt  = isset($item['alt']) ? $item['alt'] : '';
            if ($code && $url) {
                $map[$code] = ['url' => $url, 'alt' => $alt];
                // Also store base code if not explicitly provided
                $base = $this->base_lang($code);
                if ($base && !isset($map[$base])) {
                    $map[$base] = ['url' => $url, 'alt' => $alt];
                }
            }
        }

        // Determine current language from WP (server-side best effort)
        $doc_lang = get_bloginfo('language'); // e.g. fa-IR or en-US
        $doc_lang = $this->normalize_lang($doc_lang);
        $server_choice = '';
        $server_alt = '';

        if (isset($map[$doc_lang])) {
            $server_choice = $map[$doc_lang]['url'];
            $server_alt = $map[$doc_lang]['alt'];
        } else {
            $base = $this->base_lang($doc_lang);
            if (isset($map[$base])) {
                $server_choice = $map[$base]['url'];
                $server_alt = $map[$base]['alt'];
            } elseif ($default_image) {
                $server_choice = $default_image;
                $server_alt = '';
            }
        }

        // Fallback if still empty: pick first item
        if (!$server_choice && !empty($items)) {
            $first = reset($items);
            $server_choice = isset($first['image']['url']) ? $first['image']['url'] : '';
            $server_alt = isset($first['alt']) ? $first['alt'] : '';
        }

        $widget_id = esc_attr($this->get_id());
        $json_map = wp_json_encode($map);
        $json_default = wp_json_encode(['url' => $default_image]);
        ?>
        <div class="mn-language-image-wrapper" id="mn-language-image-<?php echo $widget_id; ?>"
             data-images='<?php echo esc_attr($json_map); ?>'
             data-default='<?php echo esc_attr($json_default); ?>'
             data-link="<?php echo esc_url($link_url); ?>">
            <?php if ($server_choice): ?>
                <a class="mn-lang-link" href="<?php echo esc_url($link_url); ?>"<?php echo $link_is_external ? ' target="_blank"' : ''; ?><?php echo $link_nofollow ? ' rel="nofollow"' : ''; ?>>
                    <img class="mn-lang-image" src="<?php echo esc_url($server_choice); ?>" alt="<?php echo esc_attr($server_alt); ?>" loading="lazy" />
                </a>
            <?php endif; ?>
        </div>
        <script>
        (function(){
            try {
                var el = document.getElementById('mn-language-image-<?php echo $widget_id; ?>');
                if (!el) return;
                var img = el.querySelector('img.mn-lang-image');
                var anchor = el.querySelector('a.mn-lang-link');
                var map = el.getAttribute('data-images');
                var def = el.getAttribute('data-default');
                var linkHref = el.getAttribute('data-link') || '';
                map = map ? JSON.parse(map) : {};
                def = def ? JSON.parse(def) : {url: ''};

                function pickForLang(docLang){
                    docLang = (docLang || '').toLowerCase();
                    var url = '', alt = '';
                    if (docLang && map[docLang]) {
                        url = map[docLang].url; alt = map[docLang].alt || '';
                    } else if (docLang) {
                        var base = docLang.split('-')[0];
                        if (map[base]) { url = map[base].url; alt = map[base].alt || ''; }
                    }
                    if (!url && def && def.url) url = def.url;
                    return {url:url, alt:alt};
                }

                function ensureAnchor(){
                    if (!anchor) {
                        anchor = document.createElement('a');
                        anchor.className = 'mn-lang-link';
                        if (linkHref) anchor.setAttribute('href', linkHref);
                        if (img) {
                            img.parentNode ? img.parentNode.insertBefore(anchor, img) : el.appendChild(anchor);
                            anchor.appendChild(img);
                        } else {
                            el.appendChild(anchor);
                        }
                    }
                    if (linkHref) anchor.setAttribute('href', linkHref);
                }

                function updateImage(){
                    var docLang = (document.documentElement.getAttribute('lang') || '').toLowerCase();
                    var chosen = pickForLang(docLang);
                    ensureAnchor();
                    if (!img) {
                        // Create img if not present (e.g., initially no server_choice)
                        img = document.createElement('img');
                        img.className = 'mn-lang-image';
                        img.loading = 'lazy';
                        anchor ? anchor.appendChild(img) : el.appendChild(img);
                    }
                    if (chosen.url && img.getAttribute('src') !== chosen.url) {
                        img.setAttribute('src', chosen.url);
                        if (chosen.alt) img.setAttribute('alt', chosen.alt);
                    }
                }

                // Initial update
                updateImage();

                // Observe <html> attribute changes (lang/dir)
                try {
                    var htmlObserver = new MutationObserver(function(mutations){
                        for (var i=0;i<mutations.length;i++){
                            if (mutations[i].type === 'attributes') {
                                updateImage();
                                break;
                            }
                        }
                    });
                    htmlObserver.observe(document.documentElement, { attributes: true, attributeFilter: ['lang','dir'] });
                } catch(e){}

                // Observe DOM additions similar to the provided snippet
                try {
                    var bodyObserver = new MutationObserver(function(mutations){
                        mutations.forEach(function(mutation){
                            if (mutation.addedNodes && mutation.addedNodes.length){
                                updateImage();
                            }
                        });
                    });
                    if (document.body) {
                        bodyObserver.observe(document.body, { childList: true, subtree: true });
                    } else {
                        document.addEventListener('DOMContentLoaded', function(){
                            if (document.body) bodyObserver.observe(document.body, { childList: true, subtree: true });
                        });
                    }
                } catch(e){}
            } catch(e) {
                // silent
            }
        })();
        </script>
        <?php
    }
}
