<?php
if (!defined('ABSPATH')) {
    exit;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;

class Mohtavanegar_Medpress_Gallery_Widget extends Widget_Base {
    public function get_name() {
        return 'medpress_gallery';
    }

    public function get_title() {
        return esc_html__('گالری مدپرس', 'mohtavanegar');
    }

    public function get_icon() {
        return 'eicon-gallery-grid';
    }

    public function get_categories() {
        // Put it under our custom category to sit with the rest of the theme widgets
        return [ 'mohtavanegar' ];
    }

    public function get_keywords() {
        return [ 'gallery', 'image', 'grid', 'lightbox', 'گالری', 'تصویر' ];
    }

    // Controls: build our own, similar to Elementor basic gallery
    protected function register_controls() {
        // Section: Basic Gallery
        $this->start_controls_section(
            'section_gallery',
            [
                'label' => esc_html__('Medpress Basic Gallery', 'mohtavanegar'),
            ]
        );

        $this->add_control(
            'wp_gallery',
            [
                'label' => esc_html__('Add Images', 'mohtavanegar'),
                'type' => Controls_Manager::GALLERY,
                'show_label' => false,
                'dynamic' => [ 'active' => true ],
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'exclude' => [ 'custom' ],
            ]
        );

        $gallery_columns = range(1, 10);
        $gallery_columns = array_combine($gallery_columns, $gallery_columns);

        $this->add_responsive_control(
            'gallery_columns',
            [
                'label' => esc_html__('Columns', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => $gallery_columns,
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'default' => 4,
                'tablet_default' => 3,
                'mobile_default' => 2,
                'selectors' => [
                    '{{WRAPPER}} .gallery-grid .gallery-item' => 'width: calc(100% / {{VALUE}}); flex: 0 0 calc(100% / {{VALUE}});text-align:center;',
                ],
            ]
        );

        $this->add_control(
            'gallery_display_caption',
            [
                'label' => esc_html__('Caption', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'default' => 'attachment',
                'options' => [
                    'none' => esc_html__('None', 'mohtavanegar'),
                    'attachment' => esc_html__('Attachment Caption', 'mohtavanegar'),
                    'title' => esc_html__('Image Title', 'mohtavanegar'),
                    'caption_then_title' => esc_html__('Caption then Title', 'mohtavanegar'),
                    'title_then_caption' => esc_html__('Title then Caption', 'mohtavanegar'),
                ],
                'selectors_dictionary' => [
                    'none' => 'none',
                    'attachment' => 'block',
                    'title' => 'block',
                    'caption_then_title' => 'block',
                    'title_then_caption' => 'block',
                    '' => 'block',
                ],
                'selectors' => [
                    '{{WRAPPER}} .gallery-item .gallery-caption' => 'display: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'gallery_link',
            [
                'label' => esc_html__('Link', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'default' => 'file',
                'options' => [
                    'file' => esc_html__('Media File', 'mohtavanegar'),
                    'attachment' => esc_html__('Attachment Page', 'mohtavanegar'),
                    'none' => esc_html__('None', 'mohtavanegar'),
                ],
            ]
        );

        $this->add_control(
            'open_lightbox',
            [
                'label' => esc_html__('Lightbox', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'default' => 'default',
                'options' => [
                    'default' => esc_html__('Default', 'mohtavanegar'),
                    'yes' => esc_html__('Yes', 'mohtavanegar'),
                    'no' => esc_html__('No', 'mohtavanegar'),
                ],
                'condition' => [ 'gallery_link' => 'file' ],
            ]
        );

        $this->add_control(
            'show_lightbox_caption',
            [
                'label' => esc_html__('Show Caption in Lightbox', 'mohtavanegar'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'mohtavanegar'),
                'label_off' => esc_html__('Hide', 'mohtavanegar'),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'gallery_link' => 'file',
                    'open_lightbox!' => 'no',
                ],
            ]
        );

        $this->add_control(
            'gallery_rand',
            [
                'label' => esc_html__('Order By', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => esc_html__('Default', 'mohtavanegar'),
                    'rand' => esc_html__('Random', 'mohtavanegar'),
                ],
                'default' => '',
            ]
        );

        $this->end_controls_section();

        // Section: Images Style
        $this->start_controls_section(
            'section_gallery_images',
            [
                'label' => esc_html__('Images', 'mohtavanegar'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'image_spacing',
            [
                'label' => esc_html__('Gap', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => esc_html__('Default', 'mohtavanegar'),
                    'custom' => esc_html__('Custom', 'mohtavanegar'),
                ],
                'prefix_class' => 'gallery-spacing-',
                'default' => '',
            ]
        );

        $columns_margin = is_rtl() ? '0 0 -{{SIZE}}{{UNIT}} -{{SIZE}}{{UNIT}};' : '0 -{{SIZE}}{{UNIT}} -{{SIZE}}{{UNIT}} 0;';
        $columns_padding = is_rtl() ? '0 0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}};' : '0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0;';

        $this->add_control(
            'image_spacing_custom',
            [
                'label' => esc_html__('Custom Gap', 'mohtavanegar'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'em', 'rem', 'custom' ],
                'range' => [
                    'px' => [ 'max' => 100 ],
                    'em' => [ 'max' => 10 ],
                    'rem' => [ 'max' => 10 ],
                ],
                'default' => [ 'size' => 15 ],
                'selectors' => [
                    '{{WRAPPER}} .gallery-item' => 'padding:' . $columns_padding,
                    '{{WRAPPER}} .gallery' => 'margin: ' . $columns_margin,
                ],
                'condition' => [ 'image_spacing' => 'custom' ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'image_border',
                'selector' => '{{WRAPPER}} .gallery-item img',
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
                'selectors' => [
                    '{{WRAPPER}} .gallery-item img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Section: Caption Style
        $this->start_controls_section(
            'section_caption',
            [
                'label' => esc_html__('Caption', 'mohtavanegar'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [ 'gallery_display_caption!' => 'none' ],
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label' => esc_html__('Alignment', 'mohtavanegar'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [ 'title' => esc_html__('Left', 'mohtavanegar'), 'icon' => 'eicon-text-align-left' ],
                    'center' => [ 'title' => esc_html__('Center', 'mohtavanegar'), 'icon' => 'eicon-text-align-center' ],
                    'right' => [ 'title' => esc_html__('Right', 'mohtavanegar'), 'icon' => 'eicon-text-align-right' ],
                    'justify' => [ 'title' => esc_html__('Justified', 'mohtavanegar'), 'icon' => 'eicon-text-align-justify' ],
                ],
                'default' => 'center',
                'selectors' => [ '{{WRAPPER}} .gallery-item .gallery-caption' => 'text-align: {{VALUE}};' ],
                'condition' => [ 'gallery_display_caption!' => 'none' ],
            ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => esc_html__('Text Color', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [ '{{WRAPPER}} .gallery-item .gallery-caption' => 'color: {{VALUE}};' ],
                'condition' => [ 'gallery_display_caption!' => 'none' ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'selector' => '{{WRAPPER}} .gallery-item .gallery-caption',
                'condition' => [ 'gallery_display_caption!' => 'none' ],
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'caption_shadow',
                'selector' => '{{WRAPPER}} .gallery-item .gallery-caption',
                'condition' => [ 'gallery_display_caption!' => 'none' ],
            ]
        );

        $this->add_responsive_control(
            'caption_space',
            [
                'label' => esc_html__('Spacing', 'mohtavanegar'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
                'selectors' => [ '{{WRAPPER}} .gallery-item .gallery-caption' => 'margin-block-start: {{SIZE}}{{UNIT}};' ],
                'condition' => [ 'gallery_display_caption!' => 'none' ],
            ]
        );

        $this->end_controls_section();

        // Removed Medpress-specific caption controls; unified into Caption dropdown above
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        if (empty($settings['wp_gallery'])) {
            return;
        }

        // Order images
        $images = $settings['wp_gallery'];
        if (!empty($settings['gallery_rand'])) {
            shuffle($images);
        }

        // Columns are handled via responsive CSS; no need to compute here
        $caption_source = $settings['gallery_display_caption'] ?? 'attachment';
        // Backward compatibility (old saved value could be empty string for attachment)
        if ($caption_source === '') {
            $caption_source = 'attachment';
        }

        $wrapper_classes = [ 'elementor-medpress-gallery', 'gallery' ];
        $this->add_render_attribute('wrapper', 'class', implode(' ', $wrapper_classes));

        echo '<div ' . $this->get_render_attribute_string('wrapper') . '>';
        echo '<div class="gallery-grid" style="display:flex;flex-wrap:wrap">';

        foreach ($images as $index => $item) {
            $attachment_id = $item['id'];
            if (!$attachment_id) continue;

            // Prepare metadata
            $alt = trim(get_post_meta($attachment_id, '_wp_attachment_image_alt', true));
            $title = get_the_title($attachment_id);

            // Respect the selected image size from the control (name: 'thumbnail') without relying on Elementor helper
            $selected_size = $settings['thumbnail_size'] ?? 'thumbnail';
            if ($selected_size === 'custom') {
                $dim = $settings['thumbnail_custom_dimension'] ?? null;
                $width = is_array($dim) ? (int)($dim['width'] ?? 0) : 0;
                $height = is_array($dim) ? (int)($dim['height'] ?? 0) : 0;
                $size_param = ($width > 0 || $height > 0) ? [ $width, $height ] : 'full';
            } else {
                $size_param = $selected_size;
            }

            $img_html = wp_get_attachment_image($attachment_id, $size_param, false, [
                'alt' => esc_attr($alt ?: $title),
            ]);

            $full_url = wp_get_attachment_url($attachment_id);

            // Caption logic
            $wp_caption = wp_get_attachment_caption($attachment_id);
            switch ($caption_source) {
                case 'title':
                    $final_caption = $title ?: '';
                    break;
                case 'caption_then_title':
                    $final_caption = ($wp_caption !== '' && $wp_caption !== null) ? $wp_caption : ($title ?: '');
                    break;
                case 'title_then_caption':
                    $final_caption = $title ?: ($wp_caption ?: '');
                    break;
                case 'attachment':
                default:
                    $final_caption = $wp_caption;
                    break;
            }

            // Link logic
            $link_type = $settings['gallery_link'] ?? 'file';
            $open_lightbox = $settings['open_lightbox'] ?? 'default';
            $link_open = '';
            $link_close = '';
            $link_href = '';

            if ($link_type === 'file' && $full_url) {
                $link_href = $full_url;
            } elseif ($link_type === 'attachment') {
                $link_href = get_attachment_link($attachment_id);
            }

            $lightbox_attrs = '';
            if ($link_type === 'file' && $open_lightbox !== 'no') {
                $lightbox_attrs = ' data-elementor-open-lightbox="yes" data-elementor-lightbox-slideshow="' . esc_attr($this->get_id()) . '"';
                
                // Add caption to lightbox if enabled
                if ($settings['show_lightbox_caption'] === 'yes' && !empty($final_caption)) {
                    $lightbox_attrs .= ' data-elementor-lightbox-title="' . esc_attr(wp_strip_all_tags($final_caption)) . '"';
                }
            }

            echo '<figure class="gallery-item">';
            if ($link_type !== 'none' && $link_href) {
                echo '<a href="' . esc_url($link_href) . '" class="gallery-link"' . $lightbox_attrs . '>';
            }
            echo $img_html;
            if ($link_type !== 'none' && $link_href) {
                echo '</a>';
            }

            if ($caption_source !== 'none' && $final_caption !== '' && $final_caption !== null) {
                echo '<figcaption class="gallery-caption" style="display:block">' . esc_html($final_caption) . '</figcaption>';
            }
            echo '</figure>';
        }

        echo '</div>';
        echo '</div>';
    }
}
