<?php
if (!defined('ABSPATH')) { exit; }

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

class Mohtavanegar_Search_Modal_Widget extends Widget_Base {
    public function get_name() {
        return 'search_modal';
    }

    public function get_title() {
        return esc_html__('Search Modal', 'mohtavanegar');
    }

    public function get_icon() {
        return 'eicon-search';
    }

    public function get_categories() {
        return [ 'mohtavanegar' ];
    }

    public function get_keywords() {
        return [ 'search', 'modal', 'ajax', 'popup', 'جستجو' ];
    }

    protected function register_controls() {
        // Section: Button Settings
        $this->start_controls_section(
            'section_button',
            [ 'label' => esc_html__('Button', 'mohtavanegar') ]
        );

        $this->add_control(
            'button_icon',
            [
                'label' => esc_html__('Icon', 'mohtavanegar'),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-search',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $this->add_control(
            'button_bg',
            [
                'label' => esc_html__('Background', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mn-search-modal__toggle' => 'background-color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__('Icon Color', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mn-search-modal__toggle, {{WRAPPER}} .mn-search-modal__toggle i' => 'color: {{VALUE}}; fill: {{VALUE}};'
                ],
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => esc_html__('Padding', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', 'rem' ],
                'selectors' => [
                    '{{WRAPPER}} .mn-search-modal__toggle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]
        );

        $this->add_responsive_control(
            'button_margin',
            [
                'label' => esc_html__('Margin', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', 'rem' ],
                'selectors' => [
                    '{{WRAPPER}} .mn-search-modal__toggle' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'selector' => '{{WRAPPER}} .mn-search-modal__toggle',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_shadow',
                'selector' => '{{WRAPPER}} .mn-search-modal__toggle',
            ]
        );

        $this->add_control(
            'button_border_radius',
            [
                'label' => esc_html__('Border Radius', 'mohtavanegar'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .mn-search-modal__toggle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Section: Query Settings
        $this->start_controls_section(
            'section_query',
            [ 'label' => esc_html__('Query', 'mohtavanegar') ]
        );

        // Build post types list (public only)
        $post_types = get_post_types([ 'public' => true ], 'objects');
        $options = [];
        foreach ($post_types as $pt) {
            $options[$pt->name] = $pt->labels->singular_name;
        }

        $this->add_control(
            'post_types',
            [
                'label' => esc_html__('Post Types', 'mohtavanegar'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => $options,
                'default' => array_keys($options),
            ]
        );

        $this->end_controls_section();

        // Section: Texts
        $this->start_controls_section(
            'section_texts',
            [ 'label' => esc_html__('Texts', 'mohtavanegar') ]
        );

        $this->add_control(
            'search_placeholder',
            [
                'label' => esc_html__('Search Placeholder', 'mohtavanegar'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Search...', 'mohtavanegar'),
                'placeholder' => esc_html__('Search...', 'mohtavanegar'),
            ]
        );

        $this->add_control(
            'more_results_text',
            [
                'label' => esc_html__('More Results Text', 'mohtavanegar'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('More results', 'mohtavanegar'),
                'placeholder' => esc_html__('More results', 'mohtavanegar'),
            ]
        );

        $this->end_controls_section();

        // Section: Modal Styles
        $this->start_controls_section(
            'section_modal_style',
            [ 'label' => esc_html__('Modal', 'mohtavanegar'), 'tab' => Controls_Manager::TAB_STYLE ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'input_typo',
                'selector' => '{{WRAPPER}} .mn-search-modal__input',
            ]
        );

        $this->add_control(
            'modal_bg',
            [
                'label' => esc_html__('Overlay Background', 'mohtavanegar'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mn-search-modal__overlay' => 'background-color: {{VALUE}};'
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $widget_id = $this->get_id();

        // Data attributes for JS
        $data = [
            'postTypes' => !empty($settings['post_types']) ? array_values((array)$settings['post_types']) : ['post'],
        ];

        $this->add_render_attribute('wrapper', 'class', 'mn-search-modal');
        $this->add_render_attribute('wrapper', 'data-settings', esc_attr(wp_json_encode($data)));

        echo '<div ' . $this->get_render_attribute_string('wrapper') . '>';
        echo '  <button type="button" class="mn-search-modal__toggle" aria-haspopup="dialog" aria-controls="mn-search-modal-' . esc_attr($widget_id) . '" aria-expanded="false">';
        // Icon
        if (!empty($settings['button_icon']['value'])) {
            \Elementor\Icons_Manager::render_icon($settings['button_icon'], ['aria-hidden' => 'true']);
        } else {
            echo '<span class="dashicons dashicons-search"></span>';
        }
        echo '  </button>';

        // Modal Structure
        echo '<div id="mn-search-modal-' . esc_attr($widget_id) . '" class="mn-search-modal__overlay" role="dialog" aria-modal="true" aria-hidden="true">';
        echo '  <div class="mn-search-modal__dialog" role="document">';
        echo '    <button type="button" class="mn-search-modal__close" aria-label="' . esc_attr__('Close', 'mohtavanegar') . '">&times;</button>';
        echo '    <div class="mn-search-modal__header">';
        $ph = isset($settings['search_placeholder']) && $settings['search_placeholder'] !== '' ? $settings['search_placeholder'] : esc_html__('Search...', 'mohtavanegar');
        echo '      <input type="search" class="mn-search-modal__input" placeholder="' . esc_attr($ph) . '" />';
        echo '    </div>';
        echo '    <div class="mn-search-modal__body">';
        echo '      <div class="mn-search-modal__loading" hidden>' . esc_html__('Loading...', 'mohtavanegar') . '</div>';
        echo '      <ul class="mn-search-modal__results" aria-live="polite"></ul>';
        echo '      <div class="mn-search-modal__footer">';
        $more_txt = isset($settings['more_results_text']) && $settings['more_results_text'] !== '' ? $settings['more_results_text'] : esc_html__('More results', 'mohtavanegar');
        echo '        <a class="mn-search-modal__more button" href="#" target="_self" hidden>' . esc_html($more_txt) . '</a>';
        echo '      </div>';
        echo '    </div>';
        echo '  </div>';
        echo '</div>';
        echo '</div>';
    }

    public static function ajax_search_handler() {
        check_ajax_referer('mn_search_modal_nonce', 'nonce');

        $q = isset($_POST['q']) ? sanitize_text_field(wp_unslash($_POST['q'])) : '';
        $post_types = isset($_POST['post_types']) && is_array($_POST['post_types']) ? array_map('sanitize_key', $_POST['post_types']) : ['post'];
        $per_page = 20; // return up to 20, frontend shows first 10

        $args = [
            's' => $q,
            'post_type' => $post_types,
            'post_status' => 'publish',
            'posts_per_page' => $per_page,
            'no_found_rows' => false,
        ];

        $query = new WP_Query($args);
        $items = [];

        if ($query->have_posts()) {
            foreach ($query->posts as $p) {
                $items[] = [
                    'title' => get_the_title($p),
                    'link' => get_permalink($p),
                    'excerpt' => wp_trim_words(wp_strip_all_tags(get_the_excerpt($p)), 20),
                ];
            }
        }

        wp_send_json_success([
            'total' => (int)$query->found_posts,
            'items' => $items,
        ]);
    }
}
