<?php
if (!defined('ABSPATH')) exit;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;

class Video_Gallery_Widget extends \Elementor\Widget_Base {
    public function get_name() {
        return 'video_gallery_widget';
    }

    public function get_title() {
        return __('گالری ویدئو', 'mohtavanegar');
    }

    public function get_icon() {
        return 'eicon-video-playlist';
    }

    public function get_categories() {
        return ['mohtavanegar'];
    }

    protected function _register_controls() {
        $this->start_controls_section('section_content', [
            'label' => __('محتوا', 'mohtavanegar'),
        ]);

        $this->add_control('query_source', [
            'label' => __('منبع ویدئوها', 'mohtavanegar'),
            'type' => Controls_Manager::SELECT,
            'default' => 'latest',
            'options' => [
                'latest' => __('جدیدترین ویدئوها', 'mohtavanegar'),
                'related' => __('مرتبط با پست فعلی (هوشمند)', 'mohtavanegar'),
                'manual_search' => __('جستجوی دستی (بر اساس عنوان)', 'mohtavanegar'),
                'manual_selection' => __('انتخاب دستی ویدئوها', 'mohtavanegar'),
            ],
        ]);

        $this->add_control('manual_search_term', [
            'label' => __('عنوان مورد نظر برای جستجو', 'mohtavanegar'),
            'type' => Controls_Manager::TEXT,
            'placeholder' => __('مثلاً: ارتودنسی', 'mohtavanegar'),
            'condition' => [
                'query_source' => 'manual_search',
            ],
        ]);

        $this->add_control('select_videos', [
            'label' => __('انتخاب ویدئوها', 'mohtavanegar'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => $this->get_all_videos(),
            'label_block' => true,
            'condition' => [
                'query_source' => 'manual_selection',
            ],
        ]);

        $this->add_control('items_limit', [
            'label' => __('تعداد ویدئوها', 'mohtavanegar'),
            'type' => Controls_Manager::NUMBER,
            'default' => 6,
            'condition' => [
                'query_source!' => 'manual_selection',
            ],
        ]);

        $this->add_control('select_category', [
            'label' => __('انتخاب دسته‌بندی', 'mohtavanegar'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => $this->get_video_categories(),
            'label_block' => true,
            'condition' => [
                'query_source' => 'latest',
            ],
        ]);

        $this->add_responsive_control('columns', [
            'label' => __('تعداد ستون‌ها', 'mohtavanegar'),
            'type' => Controls_Manager::SELECT,
            'default' => '3',
            'tablet_default' => '2',
            'mobile_default' => '1',
            'options' => [
                '1' => '1',
                '2' => '2',
                '3' => '3',
                '4' => '4',
            ],
            'selectors' => [
                '{{WRAPPER}} .video-gallery-grid' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
            ],
        ]);

        $this->add_control('enable_pagination', [
            'label' => __('فعال‌سازی صفحه‌بندی', 'mohtavanegar'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('بله', 'mohtavanegar'),
            'label_off' => __('خیر', 'mohtavanegar'),
            'return_value' => 'yes',
            'default' => 'no',
        ]);

        $this->add_control('show_featured_image', [
            'label' => __('نمایش تصویر شاخص', 'mohtavanegar'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('نمایش', 'mohtavanegar'),
            'label_off' => __('مخفی', 'mohtavanegar'),
            'return_value' => 'yes',
            'default' => 'yes',
        ]);

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'default' => 'medium_large',
                'separator' => 'before',
                'condition' => [
                    'show_featured_image' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section('section_style', [
            'label' => __('استایل', 'mohtavanegar'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_control('item_gap', [
            'label' => __('فاصله بین آیتم‌ها', 'mohtavanegar'),
            'type' => Controls_Manager::SLIDER,
            'range' => ['px' => ['min' => 0, 'max' => 50]],
            'default' => ['size' => 20],
            'selectors' => [
                '{{WRAPPER}} .video-gallery-grid' => 'gap: {{SIZE}}{{UNIT}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'item_box_shadow',
            'selector' => '{{WRAPPER}} .video-gallery-item',
        ]);

        $this->add_control('item_border_radius', [
            'label' => __('شعاع گوشه', 'mohtavanegar'),
            'type' => Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%'],
            'selectors' => [
                '{{WRAPPER}} .video-gallery-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]);

        $this->add_control('heading_play_button', [
            'label' => __('دکمه پخش (Play)', 'mohtavanegar'),
            'type' => Controls_Manager::HEADING,
            'separator' => 'before',
        ]);

        $this->add_control('play_bg_color', [
            'label' => __('رنگ پس‌زمینه دکمه', 'mohtavanegar'),
            'type' => Controls_Manager::COLOR,
            'default' => 'rgba(255,255,255,0.8)',
            'selectors' => [
                '{{WRAPPER}} .play-overlay' => 'background-color: {{VALUE}};',
            ],
        ]);

        $this->add_control('play_icon_color', [
            'label' => __('رنگ آیکون', 'mohtavanegar'),
            'type' => Controls_Manager::COLOR,
            'default' => '#000',
            'selectors' => [
                '{{WRAPPER}} .play-overlay i' => 'color: {{VALUE}};',
            ],
        ]);

        $this->add_control('play_size', [
            'label' => __('اندازه دکمه', 'mohtavanegar'),
            'type' => Controls_Manager::SLIDER,
            'range' => ['px' => ['min' => 30, 'max' => 150]],
            'default' => ['size' => 60],
            'selectors' => [
                '{{WRAPPER}} .play-overlay' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .play-overlay i' => 'font-size: calc({{SIZE}}{{UNIT}} / 2.5);',
            ],
        ]);

        $this->add_control('heading_title_style', [
            'label' => __('عنوان', 'mohtavanegar'),
            'type' => Controls_Manager::HEADING,
            'separator' => 'before',
        ]);

        $this->add_responsive_control('title_alignment', [
            'label' => __('تراز بندی متن', 'mohtavanegar'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => __('چپ', 'mohtavanegar'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => __('وسط', 'mohtavanegar'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => __('راست', 'mohtavanegar'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'selectors' => [
                '{{WRAPPER}} .video-info' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->add_control('title_color', [
            'label' => __('رنگ عنوان', 'mohtavanegar'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .video-info h3, {{WRAPPER}} .video-info h3 a' => 'color: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => __('تایپوگرافی', 'mohtavanegar'),
                'selector' => '{{WRAPPER}} .video-info h3, {{WRAPPER}} .video-info h3 a',
            ]
        );

        $this->add_control('title_spacing', [
            'label' => __('فاصله از تصویر', 'mohtavanegar'),
            'type' => Controls_Manager::SLIDER,
            'range' => ['px' => ['min' => 0, 'max' => 100]],
            'default' => ['size' => 15],
            'selectors' => [
                '{{WRAPPER}} .video-info' => 'padding-top: {{SIZE}}{{UNIT}};',
            ],
        ]);

        $this->add_control('heading_video_icon', [
            'label' => __('آیکون ویدئو', 'mohtavanegar'),
            'type' => Controls_Manager::HEADING,
            'separator' => 'before',
        ]);

        $this->add_control('video_icon_color', [
            'label' => __('رنگ آیکون', 'mohtavanegar'),
            'type' => Controls_Manager::COLOR,
            'default' => '#666',
            'selectors' => [
                '{{WRAPPER}} .video-info h3 .fa-video' => 'color: {{VALUE}};',
            ],
        ]);

        $this->add_control('video_icon_bg_color', [
            'label' => __('پس‌زمینه آیکون', 'mohtavanegar'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .video-info h3 .fa-video' => 'background-color: {{VALUE}};',
            ],
        ]);

        $this->add_control('video_icon_padding', [
            'label' => __('فاصله داخلی', 'mohtavanegar'),
            'type' => Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%'],
            'selectors' => [
                '{{WRAPPER}} .video-info h3 .fa-video' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]);

        $this->add_control('video_icon_border_radius', [
            'label' => __('شعاع حاشیه', 'mohtavanegar'),
            'type' => Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%'],
            'selectors' => [
                '{{WRAPPER}} .video-info h3 .fa-video' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]);

        $this->end_controls_section();
    }

    private function get_all_videos() {
        $posts = get_posts([
            'post_type' => 'video_gallery',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        $options = [];
        if (!empty($posts)) {
            foreach ($posts as $post) {
                $options[$post->ID] = $post->post_title;
            }
        }
        return $options;
    }

    private function get_video_categories() {
        if (!taxonomy_exists('video_category')) {
            return [];
        }
        $categories = get_terms([
            'taxonomy' => 'video_category',
            'hide_empty' => false,
        ]);
        $options = [];
        if (!is_wp_error($categories) && !empty($categories)) {
            foreach ($categories as $category) {
                $options[$category->term_id] = $category->name;
            }
        }
        return $options;
    }

    public static function render_video_items($query, $settings) {
        if ($query->have_posts()) {
            while ($query->have_posts()): $query->the_post(); 
                $video_url = get_post_meta(get_the_ID(), '_video_url', true);
                $subtitles = get_post_meta(get_the_ID(), '_video_subtitles', true);
                if (!is_array($subtitles)) $subtitles = array();
                ?>
                <div class="video-gallery-item" style="position:relative; overflow:hidden;">
                    <?php if ($settings['show_featured_image'] === 'yes'): ?>
                    <div class="video-thumb-container" style="position:relative; cursor:pointer;" onclick='openVideoPlayer(<?php echo json_encode(esc_url($video_url)); ?>, <?php echo json_encode($subtitles); ?>)'>
                        <?php if (has_post_thumbnail()): ?>
                            <?php 
                            $size_key = $settings['thumbnail_size'];
                            $img_style = 'width:100%; display:block;';
                            $image_size = $size_key;

                            if ('custom' === $size_key) {
                                $width = isset($settings['thumbnail_custom_dimension']['width']) ? $settings['thumbnail_custom_dimension']['width'] : '';
                                $height = isset($settings['thumbnail_custom_dimension']['height']) ? $settings['thumbnail_custom_dimension']['height'] : '';
                                $image_size = [ $width, $height ];
                                
                                $img_style .= 'height:' . ($height ? $height . 'px' : 'auto') . ';';
                                $img_style .= 'object-fit: cover;';
                            }
                            
                            the_post_thumbnail($image_size, ['style' => $img_style]);
                            ?>
                        <?php else: ?>
                            <div style="aspect-ratio:16/9; background:#eee; display:flex; align-items:center; justify-content:center;">
                                <i class="fas fa-video" style="font-size:40px; color:#ccc;"></i>
                            </div>
                        <?php endif; ?>
                        <div class="play-overlay" style="position:absolute; top:50%; left:50%; transform:translate(-50%,-50%); border-radius:50%; display:flex; align-items:center; justify-content:center; transition:0.3s;">
                            <i class="fas fa-play" style="margin-left:3px;"></i>
                        </div>
                    </div>
                    <?php endif; ?>
                    <div class="video-info" style="padding:15px; background:#fff;">
                        <h3 style="margin:0; font-size:16px; display: flex; align-items: center; gap: 8px;">
                            <i class="fas fa-video" style="color: #666; font-size: 14px;"></i>
                            <a href="<?php the_permalink(); ?>" style="text-decoration:none; color:inherit; transition:0.3s;" <?php if ($settings['show_featured_image'] !== 'yes') echo 'onclick=\'openVideoPlayer(' . json_encode(esc_url($video_url)) . ', ' . json_encode($subtitles) . ')\' style="cursor:pointer;"'; ?>>
                                <?php the_title(); ?>
                            </a>
                        </h3>
                    </div>
                </div>
            <?php endwhile;
            wp_reset_postdata();
        }
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        $args = [
            'post_type' => 'video_gallery',
            'posts_per_page' => (int) ($settings['items_limit'] ?: 6),
        ];

        if (!empty($settings['select_category']) && $settings['query_source'] === 'latest') {
            $args['tax_query'] = [
                [
                    'taxonomy' => 'video_category',
                    'field'    => 'term_id',
                    'terms'    => $settings['select_category'],
                ],
            ];
        }

        if ('related' === $settings['query_source']) {
            $current_post_id = get_the_ID();
            $current_title = get_the_title($current_post_id);
            if ($current_title) {
                // Broaden search: Split title into keywords, filter out short words
                $keywords = explode(' ', $current_title);
                $filtered_keywords = array_filter($keywords, function($word) {
                    return mb_strlen($word) > 3; // Only keep words longer than 3 characters
                });
                
                if (!empty($filtered_keywords)) {
                    $args['s'] = implode(' ', array_slice($filtered_keywords, 0, 4)); // Use first 4 significant keywords
                } else {
                    $args['s'] = $current_title;
                }

                if (get_post_type($current_post_id) === 'video_gallery') {
                    $args['post__not_in'] = [$current_post_id];
                }
            }
        } elseif ('manual_search' === $settings['query_source'] && !empty($settings['manual_search_term'])) {
            $args['s'] = $settings['manual_search_term'];
        } elseif ('manual_selection' === $settings['query_source'] && !empty($settings['select_videos'])) {
            $args['post__in'] = $settings['select_videos'];
            $args['orderby'] = 'post__in';
            unset($args['posts_per_page']); // Show all selected items
        }

        $query = new \WP_Query($args);

        if ($query->have_posts()): ?>
            <div id="video-gallery-<?php echo esc_attr($this->get_id()); ?>" 
                 class="video-gallery-wrapper" 
                 data-id="<?php echo esc_attr($this->get_id()); ?>"
                 data-settings='<?php echo json_encode($settings); ?>'>
                <div class="video-gallery-grid" style="display: grid;">
                    <?php self::render_video_items($query, $settings); ?>
                </div>
                
                <?php if ('yes' === $settings['enable_pagination'] && $query->max_num_pages > 1): ?>
                    <div class="video-gallery-pagination" style="margin-top: 30px; text-align: center; display: flex; justify-content: center; gap: 10px; flex-wrap: wrap;">
                        <?php
                        echo paginate_links(array(
                            'base'         => '#',
                            'format'       => '',
                            'current'      => 1,
                            'total'        => $query->max_num_pages,
                            'prev_text'    => '<i class="fas fa-chevron-right"></i>',
                            'next_text'    => '<i class="fas fa-chevron-left"></i>',
                            'type'         => 'plain',
                        ));
                        ?>
                    </div>
                <?php endif; ?>
                
                <div class="video-gallery-loader" style="display: none; position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255,255,255,0.7); z-index: 10; align-items: center; justify-content: center;">
                    <div class="spinner-border text-primary" role="status">
                        <span class="sr-only">Loading...</span>
                    </div>
                </div>
            </div>
            
            <?php
            // Enqueue Plyr if not already enqueued
            wp_enqueue_style('plyr-css', 'https://cdn.plyr.io/3.7.8/plyr.css', array(), '3.7.8');
            wp_enqueue_script('plyr-js', 'https://cdn.plyr.io/3.7.8/plyr.polyfilled.js', array(), '3.7.8', true);
            
            // Add inline script for openVideoPlayer function
            $video_player_script = "
            function openVideoPlayer(url, subtitles) {
                if (!url) return;
                
                let tracksHtml = '';
                if (subtitles && subtitles.length > 0) {
                    subtitles.forEach((sub, index) => {
                        tracksHtml += `<track kind=\"captions\" label=\"\${sub.lang}\" srclang=\"\${sub.lang.toLowerCase()}\" src=\"\${sub.file}\" \${index === 0 ? 'default' : ''}>`;
                    });
                }

                Swal.fire({
                    html: `
                        <div class=\"plyr__video-embed\" id=\"player-container\">
                            <video id=\"player\" playsinline controls>
                                <source src=\"\${url}\" type=\"video/mp4\">
                                \${tracksHtml}
                            </video>
                        </div>
                    `,
                    showConfirmButton: false,
                    showCloseButton: true,
                    width: '800px',
                    background: '#000',
                    padding: '0',
                    didOpen: () => {
                        if (typeof Plyr !== 'undefined') {
                            const player = new Plyr('#player', {
                                captions: { active: true, update: true, language: 'auto' },
                                autoplay: true
                            });
                            player.on('ready', event => {
                                player.play();
                            });
                        } else {
                            console.error('Plyr is not defined in Swal didOpen');
                        }
                    }
                });
            }
            ";
            wp_add_inline_script('plyr-js', $video_player_script);
            ?>

            <style>
            .video-gallery-item:hover .play-overlay {
                background: #fff;
                transform: translate(-50%,-50%) scale(1.1);
            }
            .swal2-html-container {
                overflow: hidden !important;
                border-radius: 8px;
            }
            :root {
                --plyr-color-main: #3498db;
            }
            .video-gallery-wrapper {
                position: relative;
            }
            .video-gallery-pagination a, .video-gallery-pagination span {
                display: inline-block;
                padding: 10px 15px;
                border-radius: 5px;
                background: #f4f4f4;
                color: #333;
                text-decoration: none;
                transition: 0.3s;
                font-weight: bold;
                min-width: 40px;
                text-align: center;
            }
            .video-gallery-pagination span.current {
                background: var(--plyr-color-main);
                color: #fff;
            }
            .video-gallery-pagination a:hover {
                background: #e0e0e0;
            }
            .video-gallery-loader {
                display: none;
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(255,255,255,0.7);
                z-index: 10;
                align-items: center;
                justify-content: center;
            }
            .spinner-border {
                display: inline-block;
                width: 2rem;
                height: 2rem;
                vertical-align: text-bottom;
                border: .25em solid currentColor;
                border-right-color: transparent;
                border-radius: 50%;
                animation: spinner-border .75s linear infinite;
            }
            @keyframes spinner-border {
                to { transform: rotate(360deg); }
            }
            </style>
            <script>
            jQuery(document).ready(function($) {
                $(document).on('click', '.video-gallery-pagination a', function(e) {
                    e.preventDefault();
                    let $container = $(this).closest('.video-gallery-wrapper');
                    let widgetId = $container.data('id');
                    let settings = $container.data('settings');
                    let page = 1;
                    
                    let href = $(this).attr('href');
                    if (href && href.indexOf('paged=') !== -1) {
                        page = href.split('paged=')[1].split('&')[0];
                    } else {
                        // Handle numeric text for current page if href is just #
                        let text = $(this).text();
                        if (!isNaN(text)) {
                            page = parseInt(text);
                        } else if ($(this).hasClass('prev')) {
                            page = parseInt($container.find('.pagination span.current').text()) - 1;
                        } else if ($(this).hasClass('next')) {
                            page = parseInt($container.find('.pagination span.current').text()) + 1;
                        }
                    }
                    
                    // Elementor use # with special format sometimes, but paginate_links with base '#' usually produces links like #page/2 or similar if configured
                    // But here we use a simple approach: find the page number from the link text or data
                    if ($(this).hasClass('page-numbers')) {
                        let text = $(this).text();
                        if (!isNaN(text)) {
                            page = parseInt(text);
                        } else if ($(this).hasClass('prev')) {
                            let current = parseInt($container.find('.video-gallery-pagination span.current').text());
                            page = current - 1;
                        } else if ($(this).hasClass('next')) {
                            let current = parseInt($container.find('.video-gallery-pagination span.current').text());
                            page = current + 1;
                        }
                    }

                    if (!page || page < 1) return;

                    $container.find('.video-gallery-loader').css('display', 'flex');
                    
                    $.ajax({
                        url: '<?php echo admin_url('admin-ajax.php'); ?>',
                        type: 'POST',
                        data: {
                            action: 'mohtavanegar_video_gallery_pagination',
                            page: page,
                            settings: settings,
                            widget_id: widgetId,
                            current_post_id: '<?php echo get_the_ID(); ?>',
                            nonce: '<?php echo wp_create_nonce("video_gallery_nonce"); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $container.find('.video-gallery-grid').html(response.data.html);
                                $container.find('.video-gallery-pagination').html(response.data.pagination);
                                
                                $('html, body').animate({
                                    scrollTop: $container.offset().top - 100
                                }, 500);
                            }
                            $container.find('.video-gallery-loader').hide();
                        },
                        error: function() {
                            $container.find('.video-gallery-loader').hide();
                        }
                    });
                });
            });
            </script>
        <?php
        wp_reset_postdata();
        else:
            if ('related' !== $settings['query_source']) {
                echo '<p>' . __('ویدئویی یافت نشد.', 'mohtavanegar') . '</p>';
            }
        endif;
    }
}
